/*
  Bot AI for Bombing Run. 
  Author Mark Caldwell aka W@rHe@d of The Reliquary
*/

class UTBRSquadAI extends UTSquadAI;

var UTBRGame BRGame;
var float LastSeeEnemyRunner;
var UTBRBall TheBall;
var UTBRGoal FriendlyGoal, EnemyGoal;
var UTBRObjectiveInfo FriendlyGoalInfo, EnemyGoalInfo;
var array<AlternateRoute> EnemyGoalRoutes;
var array<AlternateRoute> FriendlyGoalRoutes;


var string reason;

struct RatingStruct
{
    var NavigationPoint Target;
    var float Rating;
};

function Init()
{
    FriendlyGoalInfo = UTBRGame(WorldInfo.Game).GetObjectiveInfo(FriendlyGoal);
    EnemyGoalInfo = UTBRGame(WorldInfo.Game).GetObjectiveInfo(EnemyGoal);
    BRGame = UTBRGame(WorldInfo.Game);
}

/*
call to make bot evaluate it's tasks. 
call when not sure if bot should do something else or not.
bot will repath if needed but otherwise will not disturb it's current path.
delegates to playerManager for control by br.
*/
function Retask(UTBot B)
{
    local UTBRPlayerManager playerManager;
    
    playerManager = GetPlayerManager(B);
    if (playerManager == none)
    {
        ShowPath(B, "Not retasking because playermanager not found");    
        return;
    }
      
    playerManager.Retask();
}

/*
entry point for AI.
delegate out to player manager so process can be controlled.
*/
function bool AssignSquadResponsibility(UTBot B)
{
    local UTBRPlayerManager playerManager;
    
    playerManager = GetPlayerManager(B);
    if (playerManager != none)
    {   
        playerManager.AssignSquadResponsibility();      
    }

    //always return true since AssignSquadResponsibilityCallBack is put on a tick.
    //we don't need any ut functionality which would result from returning false. 
    return true;     
}

//called by UTBRPlayerManager
function AssignSquadResponsibilityCallBack(UTBot B)
{
   super.AssignSquadResponsibility(B);
   
   if (UTBRGame(WorldInfo.Game).Test_ShowPath)
   {
       ShowRouteCache(GetPlayerManager(B));         
   }  
}

/*
rewritten from base class for BR.
there was no choice but to rewrite this whole routine.
in base class the main ai for grabbing vehicles and pickups are in here with the
code which makes bot seek objective. the seek objective code is of
course after the vehicle/pickup code as it should be but we are not
given any hook whatsoever into that code, such as a routine SeekObjective().
so if we were to put our own BR seek objective code before or after the
call to CheckSquadObjectives, then that overrides the vehicle/pickup code 
and bots would have trouble grabbing these things.
this causes major problems in all UT3 ai. Play CTF Coret, get the ball and
stand in a hall. Then do killbots and addbots to restart the bots, or just wait
for bots to expend their ammo or die.
Bots coming to your defense and bots attacking you come with assault rifles
and impact hammers. unless you are close to a weapon pickup they will not get any weapons.
this is because of the above issue. CTF ai does things like make bots seek the flag holder
which then short circuts the pickup ai in UTSquadAI.CheckSquadObjectives.
 
the only thing we can do is put the proper hook call into CheckSquadObjectives,
which we now call SeekObjective(). There was no centralized routine for objective seeking 
which makes it difficult for a game mod to change a bot's behavior, so now
all objective seeking logic is centralized in SeekObjective().

many other issues found. for example, covering bot would only run back and forth 
between two points, then eventually it'd say 'heading for the super health' but
never go for the pickup. holding position bots would just crowd together in one place.
bots can get stuck in loops going back and forth.
so many issues found just ended up rewriting a lot of the ut ai.
*/
function bool CheckSquadObjectives(UTBot B)
{
    local UTBRPlayerManager playerManager;
       
    /*
    with all the various algorithms broken out into routines pertaining to them,
    CheckSquadObjectives breaks down to a small chunk of code :)
    
    a simple order then becomes clear. bot goes through a basic list of tasks to perform and performs
    the first applicable task. certain tasks should be done or prefered before others, so vehicles are checked
    before weapons, and so on.
    
    note that getting regular pickups (like a health vial) are not in below list.
    this is because they get picked up along the way of doing some other task.
    for example, if bot is defending base all pickups and path nodes are used as
    defense points so bot will gather pickups automatically.
    
    when on route to objective, the ut3 engine automatically take very short detours 
    to these pickups while keeping them on route. this is why when watching a bot run
    to a base you might see it take a few steps to the side to grab a pickup.
    ut3 engine will also route bot to pickup if that pickup is along a direct route
    to objective so bot doesn't even need to take any short detour.
    
    so regular pickups are everywhere and engine will make sure bot gets them as it passes
    by them. other pickups like vehicles may not be so available and require a special 
    journey to get them, and below code will make sure that happens when needed.
    */    
    
    if ((B == none) || (B.Pawn == none))
    {
        return false;
    }

    playerManager = GetPlayerManager(B);       
    if (playerManager == none)
    {
        return false;
    }
    playerManager.SetVars();
    
    playerManager.CheckSoak = true;
       
    if (BRCheckSquadObjectives(B))
    {
        ShowPath(B, "Performing special task");
        
        return true;
    }
   
    /*
    note on AllowDetour and GettingWeapon, GettingVehicle, etc, found in below routines.
    bot should not seek pickups in certain situations, so AllowDetour checks this.
    however, situation can keep changing and the result can be a bot looping back and forth.
    the Getting variables makes the bot stay on task and not loop.

    take this example: ball is dropped in friendly base and defending bot
    which is near to ball runs out of ammo. bot should go get ball 
    instead of going after weapon. however, if while running with ball
    bot gets far enough away from base to safely find a weapon and starts looking,
    and that path takes it back towards base, then the logic to take ball away would
    kick in again and make bot loop back and forth between those two tasks.
    */
               
    if (CheckForVehicle(playerManager))
    {
        return true;
    }
    else
    {
        playerManager.VehicleGoal = none;
    }
    
    if (CheckForWeapon(playerManager))
    {
        return true;
    }    
          
    if (CheckForSuperPickup(playerManager))
    {
        return true;
    }
    
    if (CheckForEnemyPursuit(playerManager))
    {
        return true;
    }
       
    return SeekObjective(playerManager);
}


//make bot consider fight enemy exlcusively.
//instead of heading to objective they will chase enemy.
//return true if doing so.
function bool CheckForEnemyPursuit(UTBRPlayerManager playerManager)
{
    local UTBot B;
    
    B = playerManager.Bot;
    
    if ( (B.Enemy != None) && (! ShouldSeekObjective(playerManager)))    
    {
        if ( B.LostContact(5) )
            B.LoseEnemy();
        if ( B.Enemy != None )
        {
            if (B.LineOfSightTo(B.Enemy) || (WorldInfo.TimeSeconds - B.LastSeenTime < 3))
            {
                ShowPath(B, "Fight/Seek enemy");
                           
                B.FightEnemy(false, 0);
                playerManager.CheckSoak = false;
                return true;
            }
        }
    }
    
    return false;
}

//make bot consider getting vehicle. returns true if getting vehicle.
function bool CheckForVehicle(UTBRPlayerManager playerManager)
{
    local UTBot B;
    local bool ret;

    
    B = playerManager.Bot;
    
    ret = true;
       
    if (! playerManager.GettingVehicle)
    {
        if (! AllowDetour(playerManager))
        {
           ret = false;
        }
        
        if (playerManager.GettingWeapon || playerManager.GettingSuperPickup)
        {
            ret = false;
        }
        
        if (
            (UTOnslaughtNodeObjective(playerManager.FormationCenter) != none) &&
            NodeNeedsAttention(UTOnslaughtNodeObjective(playerManager.FormationCenter), playerManager) &&
            UTOnslaughtNodeObjective(playerManager.FormationCenter).BotNearObjective(playerManager.Bot)        
            )
        {
            ret = false;
        }
    }
    
    
    if (ret)
    {
        if ((UTVehicle(playerManager.PathTarget) != none) && 
            ((playerManager.FailPathTo == playerManager.PathTarget) || (playerManager.FailedTarget == playerManager.PathTarget)))
        {
            //call direct to fail path and invoke alternate pathing logic
            ret = FindPathToObjective(B, playerManager.PathTarget);
        }
        else
        {    
            ret = CheckVehicle(B);
            
            if (ret && (UTVehicle(B.RouteGoal) != none) && (! UTVehicle(B.RouteGoal).CanEnterVehicle(B.Pawn)))
            {
                ShowPath(B, "Bot tried for vehicle " $ UTVehicle(B.RouteGoal) $ " that it couldn't enter");
                ret = false;
            }
        }
    }
          
    if (! ret)
    {
        //if bot stops going to vehicle must force bot to not retry or it might get into loop 
        
        if (UTVehicle(playerManager.PathTarget) != none)
        {
            UTVehicle(playerManager.PathTarget).VehicleLostTime = WorldInfo.TimeSeconds + 20;        
        } 
    }
    
    if (ret && (playerManager.VehicleGoal != none) && (playerManager.VehicleGoal != B.RouteGoal))
    {
        //if bot goes to another vehicle then must force bot to not retry old vehicle or it might get into loop
        playerManager.VehicleGoal.VehicleLostTime = WorldInfo.TimeSeconds + 20;     
    }    
        
    if (ret)
    {
        playerManager.VehicleGoal = UTVehicle(B.RouteGoal);
        playerManager.SetPathTarget(playerManager.VehicleGoal);
        ShowPath(B, "Get vehicle " $ playerManager.PathTarget);
        playerManager.GettingVehicle = true;                           
        return true;
    }
    
    playerManager.GettingVehicle = false;
    
    return false;
}

//make bot consider getting weapon. returns true if getting weapon.
function bool CheckForWeapon(UTBRPlayerManager playerManager)
{
    local UTBot B;
    
    /* 
        note on this code from UTSquadAI.CheckSquadObjectives()
        
        getting gun must be highest priority except for ball runner.
        otherwise bots end up chasing you with impact hammers.
        important to pass 0 to FindInventoryGoal to grab anything available.
        
        doing the below call results in bots not getting weapons and running around with 
        assault rifles and impact hammers.
        the below call short circuts FindInventoryGoal(0).
        looking at the code for FindInventoryGoal, it does a (LastSearchWeight >= BestWeight) call
        and returns false because of the last 0.0004 failed call. 
    
        if (WorldInfo.TimeSeconds - B.Pawn.CreationTime < 5.0 && B.NeedWeapon() && B.FindInventoryGoal(0.0004))
        {
            B.GoalString = "Need weapon or ammo";
            B.NoVehicleGoal = B.RouteGoal;
            B.SetAttractionState();
            return true;
        }
    */
    
    /*
    uses custom weapon finding logic because ut's logic has issues.
    ut's logic can result in bot finding regular pickup instead of weapon.
    for example, a shield belt will commonly rate higher than weapon, and
    so when bot needs weapon they go off on a tangent looking for the belt
    rather than getting reloaded and getting back to kicking ass.
    if weapon can't be found, such as can be case for arena/replacement mods,
    faulty weapon mods, etc, then bot would just wander looking for regular
    pickups. pickup ratings can change in route making bot change it's course
    to another pickup which can make bot wander too much.

    this custom logic will seek only weapons, not seek anything if weapon not found,
    and keep bot on course to it's originally desired weapon. also rates things
    differently to make bot most efficient at getting reloaded and back to the fray.    
    */

    B = playerManager.Bot;
    
    if ((! playerManager.GettingWeapon) && (! AllowDetour(playerManager)))
    {
        return false;
    }

    if (playerManager.PathTarget == none) 
    {
        playerManager.GettingWeapon = false;
    }
        
    if ((playerManager.GettingWeapon) && 
        (playerManager.PathTarget != none) && 
        (playerManager.ReachedTarget(playerManager.PathTarget))
        )
    {
        playerManager.GettingWeapon = false;
    }
         
    if ((! playerManager.GettingWeapon) && playerManager.NeedWeapon() && FindWeapon(playerManager))
    {
        playerManager.GettingWeapon = true;
    }
    
    if (playerManager.GettingWeapon)
    {
        ShowPath(B, "Get weapon " $ playerManager.PathTarget);
        
        if (! FindPathToObjective(B, playerManager.PathTarget))
        {
            playerManager.GettingWeapon = false;
            Retask(B);
        }
        
        return true;
    }   
       
    return false;
}

//return true if can find a weapon/ammo near bot.
//sets playerManager.PathTarget to desired pickup.
function bool FindWeapon(UTBRPlayerManager playerManager)
{
    local UTBot B;
    local float bestRating;
    local Actor best;
    local UTPickupFactory n;
    
    
    B = playerManager.Bot;
    
    foreach WorldInfo.RadiusNavigationPoints(class'UTPickupFactory', n, B.Pawn.Location, 6000)
    {
        RateWeapon(playerManager, n, best, bestRating);
    }
    
    if (best != none)
    {
        playerManager.SetPathTarget(best);       

        return true;
    }
    
    return false;
}

//rate weapon n and set best if better than current best.
function RateWeapon(UTBRPlayerManager playerManager, NavigationPoint n, out Actor best, out float bestRating)
{
    local float rating, dist;
    local int i;
    local UTBot B;
    
    B = playerManager.Bot;
    
    if (UTAmmoPickupFactory(n) != none)
    {
        rating = UTAmmoPickupFactory(n).BotDesireability(B.Pawn, B);
        
        //ut rates ammo too low. if bot has weapon needing ammo then that
        //ammo should have higher priority. ammo is usually easier to find
        //than weapons so this results in bot getting loaded up more quickly.
        if (rating > 0)
        {
            rating += 0.8;
        }
    }
    
    if (UTWeaponPickupFactory(n) != none)
    {
        rating = B.RatePickup(n, UTWeaponPickupFactory(n).WeaponPickupClass);
    }
    
    if (UTWeaponLocker(n) != none)
    {
        rating = B.RatePickup(n, none);    
    }
    
    if (rating == 0)
    {
        return;
    }

    if (n == playerManager.FailedTarget)
    {
        //if bot had just recently failed to reach this weapon, don't retry or will get in loop
        return;
    }
    
    if (! CanSpecialReach(playerManager, n))
    {
        return;
    }
    
    if (! PickupFactory(n).ReadyToPickup(2))
    {
        return;
    }

    dist = DistanceBetweenPoints(B.Pawn.Location, n.Location);
    if (B.RouteDist > dist)
    {
        dist = B.RouteDist;
    }
    
    i = 20 - (dist / 500);
    
    rating = rating + i;

    //want to make sure bot prefers weapons in the immediate area.
    //no one likes a bot on a hike.    
    if (dist <= 1500)
    {
        rating += 2.0;
    }
               
    if (rating > bestRating)
    {
        bestRating = rating;
        best = n;
    }
}

//make bot consider getting super pickup. return true if getting super pickup.
function bool CheckForSuperPickup(UTBRPlayerManager playerManager)
{
    local bool bCheckSuperPickups, bMovingToSuperPickup;          
    local float SuperDist;
    local Vehicle V;
    local UTBot B;    
       
    if ((! playerManager.GettingSuperPickup) && (! AllowDetour(playerManager)))
    {
       return false;
    }

    //keeps bot from getting in loop failing to reach pickup    
    if (WorldInfo.TimeSeconds < playerManager.SuperPickupTime)
    {
        return false;
    } 
    
    B = playerManager.Bot;
    
    V = Vehicle(B.Pawn);

    if (B.Pawn.bCanPickupInventory)
    {
        bCheckSuperPickups = true;
    }
    else if (V != None && V.Driver != None && V.Driver.bCanPickupInventory && (UTVehicle(V) == None || !UTVehicle(V).bKeyVehicle))
    {
        bCheckSuperPickups = true;
        B.bCheckDriverPickups = true;
    }
        
    if (bCheckSuperPickups)
    {       
        if (UTHoldSpot(B.DefensePoint) != None || PriorityObjective(B) > 0)
        {
            SuperDist = 800.0;
        }
        else if ((GetOrders() == 'Freelance' || bFreelanceAttack || bFreelanceDefend) && !B.HasTimedPowerup())
        {
            //@todo: check performance - might need to clamp this
            SuperDist = class'NavigationPoint'.const.INFINITE_PATH_COST;
        }
        else if (CurrentOrders == 'Attack')
        {
            SuperDist = (B == SquadLeader && B.Skill >= 4.0) ? 6000.0 : 3000.0;
        }
        else if (CurrentOrders == 'Defend' && B.Enemy != None)
        {
            SuperDist = 1200.0;
        }
        else
        {
            SuperDist = 3200.0;
        }
        
        if (GetBall(B.Pawn) != none)
        {
            SuperDist = 800;
        }
        
        bMovingToSuperPickup = ( (PickupFactory(B.RouteGoal) != None)
                        && PickupFactory(B.RouteGoal).bIsSuperItem
                        && (B.RouteDist < 1.1*SuperDist)
                        &&  PickupFactory(B.RouteGoal).ReadyToPickup(2)
                        && (B.RatePickup(B.RouteGoal, PickupFactory(B.RouteGoal).InventoryType) > 0) );
                        
        if ( bMovingToSuperPickup || (B.Pawn.ValidAnchor() && CheckSuperItem(B, SuperDist)) )
        {
            if (PickupFactory(B.RouteGoal) == none)
            {
                return false;
            }

            playerManager.SetPathTarget(B.RouteGoal);
                                       
            ShowPath(B, "Get super pickup " $ playerManager.PathTarget);
               
            B.bCheckDriverPickups = false;
            B.GoalString = "Get super item" @ B.RouteGoal;

            if ( V != None && !V.bCanPickupInventory && (B.Pawn.Anchor == None || !B.Pawn.Anchor.bFlyingPreferred) &&
                (B.MoveTarget == B.RouteGoal || (B.RouteCache.length > 1 && B.RouteCache[1] == B.RouteGoal)) )
            {
                // get out of vehicle here so driver can get it
                if (UTVehicle(V) != None)
                {
                    UTVehicle(V).VehicleLostTime = WorldInfo.TimeSeconds + 5.0;
                }
                B.NoVehicleGoal = B.RouteGoal;
                B.LeaveVehicle(true);
            }
            else
            {
                if (! FindPathToObjective(B, playerManager.PathTarget))
                {
                    //failed to get it to stop bot from getting super pickups for a little while (or bot could loop)
                    playerManager.SuperPickupTime = WorldInfo.TimeSeconds + 15;
                    return false;
                }
                B.RouteGoal = playerManager.PathTarget;

                //B.SetAttractionState();
            }
            
            playerManager.GettingSuperPickup = true;
            
            return true;
        }
          
        B.bCheckDriverPickups = false;
    }
    
    B.bCheckDriverPickups = false;        

    playerManager.GettingSuperPickup = false;
        
    return false;
}

function ShowRouteCache(UTBRPlayerManager playerManager, optional bool cacheOnly)
{
    local int i;
    local string s;
    local UTBot B;

    B = playerManager.Bot;
    
    if (! cacheOnly)
    {
        SendBotDebugMessage(B, 
         "Orders=" $ GetOrders() $ " FormationCenter=" $ FormationCenter(B) $ 
         " DefensivePosition=" $ B.DefensivePosition      
        );
    }
        
    s = "";
    for (i = 0; i < B.RouteCache.Length; i++)
    {
        s = s $ "[" $ i $ "]" $ B.RouteCache[i] $ ",";
    }

    if (! cacheOnly)
    { 
        s ="MoveTarget=" $ playerManager.MoveTarget $ 
           " PathTarget=" $ playerManager.PathTarget $
           " PathSubTarget=" $ playerManager.PathSubTarget $ 
           " SpecialNavigationPathTarget=" $ playerManager.SpecialNavigationPathTarget $ 
           " UTBotMoveTarget=" $ B.MoveTarget $ 
           " RouteGoal=" $ B.RouteGoal $ 
           " RouteCache: " $ s;
    }
    
    SendBotDebugMessage(B, s);
}

//in BR we only use B.DefensePoint for 'Hold this position' command.
//however, we do use a UTDefensePoint actor.
//B.DefensivePosition may get set to the defense point based on a random weight.
//this allows bots to have a tendancy to use UTDefensePoints, but also to be 
//able to defend from other places. We don't want bots always at a UTDefensePoint.
//That's too predictable and makes killing them easy.
function SetDefenseScriptFor(UTBot B)
{
}

//return true if bot should push towards objective even if they see enemy.
//bot can still fight enemy but keep moving towards objective, or
//can temporarily stop moving to objective and chase down enemy.
function bool ShouldSeekObjective(UTBRPlayerManager playerManager)
{
    local UTBot B;
    
    B = playerManager.Bot;

    if (UTOnslaughtNodeObjective(playerManager.FormationCenter) != none)
    {
        return true;
    }
        
    if ( UTHoldSpot(B.DefensePoint) != None )
    {
        //allow 'Hold this position' bots to always chase enemy
        return (B.Enemy == none);       
    }
    
    if ((playermanager.FormationCenter == TheBall) && 
        ((! TheBall.IsControlledByTeamOf(B.pawn)) || (TheBall.LastHolder == B.Pawn))
       )
    {
        //always get ball
        return true;
    }
    
    if (playerManager.HorizontalDistanceFromFormationCenter > playerManager.MaxFormationCenterDefenseDistance)
    {
        //allow bot to chase enemy within their allowed distance from objective
        return true;
    }

    if ((GetOrders() == 'Attack') &&
         TheBall.IsHeldByteamOf(B.Pawn) &&
         (GetBall(B.Pawn) == none) && 
        (playerManager.Bot.LineOfSightTo(TheBall.Position())))
    {
        //if attacking bot sees it's team's runner then should move to base to clear it or receive pass 
        return true;
    } 
    
    if (GetBall(B.Pawn) != none)
    {
        //runner always seeks objective
        return true;
    }
    
    if (B.Enemy != none)
    {          
        return false;
    }
    
    return true;
}

function bool SeekObjective(UTBRPlayerManager playerManager)
{
    local UTBot B;
    local UTOnslaughtNodeObjective node;
    local bool ret;
    
    B = playerManager.Bot;

    B.GoalString = "Seek Objective";

    if ((playerManager.HorizontalDistanceFromFormationCenter <= playerManager.MaxFormationCenterDefenseDistance)
        && (!B.bInitLifeMessage))
    {
        B.bInitLifeMessage = true;
        B.SendMessage(None, 'INPOSITION', 10);
    }
    
    playerManager.CheckFormationCenterChanged();
    playerManager.CheckFormationCenterMoved();    
    
    EvaluateDefensivePosition(playerManager);
    
    if (ShouldDestroyTranslocator(B))
    {
        ShowPath(B, "Destroy translocator");
        Retask(B);
        return true;
    }
    
    if (ShouldUndeployVehicle(B))
    {
        return true;
    }               
    
    playerManager.SeekDirect = false;
              
    if ( B.IsSniping() )
    {
       playerManager.CheckSoak = false;
       return true;
    }

    if (B.Pawn.bStationary || (UTVehicle(B.Pawn) != None && UTVehicle(B.Pawn).bIsOnTrack))
    {
        if (playerManager.HorizontalDistanceFromFormationCenter <= playerManager.MaxFormationCenterDefenseDistance)
        {
            playerManager.CheckSoak = false;        
            return true;
        }
        
        B.LeaveVehicle(true);        
    }

    if (((GetOrders() == 'Attack') || (GetOrders() == 'Freelance')) && (playerManager.TakingDefensivePathTime > 0))
    {          
       //after spawning an attacking bot will take a path which is between the home goal and the ball
       if (((WorldInfo.TimeSeconds - playerManager.TakingDefensivePathTime) < 5) && 
           (HorizontalDistanceBetween(B.Pawn,FriendlyGoal) >= 4000))            
       {
           if (GotoDefensivePath(playerManager))               
           {             
                return true;
           }
       }
    }
    playerManager.TakingDefensivePathTime = 0;
                
        
    if (
         playerManager.ShouldPickupBall() ||
         ((GetBall(B.Pawn) != none) && (UTBRGoal(playerManager.FormationCenter) != none)) ||
         (UTOnslaughtNodeObjective(playerManager.FormationCenter) != none) ||
         playerManager.DoingBallReset
       )
    {    
        //go directly to objective

        playerManager.SeekDirect = true;
              
        if ((B.DefensivePosition == none) && FindPathToObjective(B, playerManager.FormationCenterTarget))
        {           
            node = UTOnslaughtNodeObjective(playerManager.FormationCenter);
            
            if (node != none)
            {
                if (node.IsNeutral() && (UTVehicle(B.Pawn) != none) && (DistanceBetween(B.Pawn, node) <= 400))
                {
                    UTVehicle(B.Pawn).VehicleLostTime = WorldInfo.TimeSeconds + 5.0;                 
                    B.LeaveVehicle(true);                  
                }
                               
            	if ((! node.IsNeutral()) && node.BotNearObjective(B))
                {
                    playerManager.CheckSoak = false;

                    ret = false;
                    
        		    if (Node.DefenderTeamIndex == B.GetTeamNum())
          		    { 
            		    ret = TellBotHowToHeal(node, B);        		   
                    }
                    else
        		    {
        		        ret = TellBotHowToDisable(node, B);
        		    }
                    
                    if (! ret)
                    {
                        //give up on node for the moment and do something else
                        ShowPath(B, "Giving up on node " $ node.name $ "  " $ B.GoalString);
                        playerManager.FormationCenter = none;
                        Retask(B);
                        return false;
                    }
                }
        	}
            
            return true;
        }           
    }
           
    if (SeekDefensivePosition(playerManager))
    {
        return true;
    }
    
    return FindPathToObjective(B, playerManager.FormationCenterTarget);
}

//had to write this (borrowed from UTOnslaughtNodeObjective)
//because routine in UTOnslaughtPowernode doesn't work for BR
function bool TellBotHowToDisable(UTOnslaughtNodeObjective node, UTBot B)
{
	local UTVehicle VehicleEnemy;

	if (node.DefenderTeamIndex == B.Squad.Team.TeamIndex)
		return false;
	if (! node.PoweredBy(B.Squad.Team.TeamIndex))
	{
		if (B.CanAttack(node))
			return false;
		else
			return B.Squad.FindPathToObjective(B, node);
	}

	//take out defensive turrets first
	VehicleEnemy = UTVehicle(B.Enemy);
	if ( VehicleEnemy != None && (VehicleEnemy.bStationary || VehicleEnemy.bIsOnTrack) &&
		(VehicleEnemy.AIPurpose == AIP_Defensive || VehicleEnemy.AIPurpose == AIP_Any) && B.LineOfSightTo(B.Enemy) )
	{
		return false;
	}

	if ( node.StandGuard(B) )
		return node.TooClose(B);

	if ( !B.Pawn.bStationary && B.Pawn.TooCloseToAttack(node) )
	{
		B.GoalString = "Back off from objective";
		B.RouteGoal = B.FindRandomDest();
		B.MoveTarget = B.RouteCache[0];
		B.SetAttractionState();
		return true;
	}
	else if ( B.CanAttack(node) )
	{
		if (node.KillEnemyFirst(B))
			return false;

		B.GoalString = "Attack Objective";
		B.DoRangedAttackOn(node);
		return true;
	}
	//MarkShootSpotsFor(B.Pawn);
	//return Super.TellBotHowToDisable(B);
	
	if (GetBall(B.Pawn) != none)
	{
	    return true;
    }
	    
	return false;
}

//from UTOnslaughtPowernode
function bool TellBotHowToHeal(UTOnslaughtNodeObjective node, UTBot B)
{
	local vector AdjustedLoc;
	local UTVehicle BotVehicle;
	local UTWeapon w; 

	// if bot is in important vehicle, don't get out
	BotVehicle = UTVehicle(B.Pawn);
	if (BotVehicle != None && BotVehicle.ImportantVehicle())
	{
		return node.TooClose(B);
	}

    if (UTOnslaughtPowernode(node) != none)
    {
    	AdjustedLoc = UTOnslaughtPowernode(node).EnergySphere.GetPosition();
    	AdjustedLoc.Z = B.Pawn.Location.Z;
    	if ( VSize(AdjustedLoc - B.Pawn.Location) < 50 )
    	{
    		// standing right on top of it, move away a little
    		B.GoalString = "Move away from "$node;
    		B.RouteGoal = B.FindRandomDest();
    		B.MoveTarget = B.RouteCache[0];
    		B.SetAttractionState();
    		return true;
    	}
    	else if (VSize(UTOnslaughtPowernode(node).EnergySphere.GetPosition() - B.Pawn.Location) > class'UTWeap_LinkGun'.default.WeaponRange)
    	{
    		// too far to heal
    		B.GoalString = "Move closer to "$node;
    		if ( B.FindBestPathToward(node, false, true) )
    		{
    			B.SetAttractionState();
    			return true;
    		}
    		else
    			return false;
    	}
	}

	if (!node.TeamLink(B.GetTeamNum()) || node.Health >= node.DamageCapacity)
	{
		if ( (B.Enemy == None) && B.PlayerReplicationInfo.bHasFlag )
		{
			// defend with flag
			B.MoveToDefensePoint();
			return true;
		}
		return false;
	}

	if (BotVehicle != None && !BotVehicle.bKeyVehicle && (B.Enemy == None || (!B.LineOfSightTo(B.Enemy) && WorldInfo.TimeSeconds - B.LastSeenTime > 3)))
	{
		B.LeaveVehicle(true);
		return true;
	}
		
	if ((UTWeapon(B.Pawn.Weapon) == None) || (! UTWeapon(B.Pawn.Weapon).CanHeal(node)))
	{
        //link gun code looks to see of Squad.SquadObjective is a power node, which is
        //a horrible algorithm because it limits the whole squad to one node.
        //so, instead we'll up the gun's rating and then restore it in FormationCenter()
        //when done with node
        
        ForEach B.Pawn.InvManager.InventoryActors(class'UTWeapon', w)
        {
            if (w.CanHeal(node))
            {
           	    w.AIRating = 1.2;
            }
    	}
	
   		B.SwitchToBestWeapon();

        return true;
    }        
    
	if (!B.Pawn.CanAttack(node))
	{
		// need to move to somewhere else near objective
		B.GoalString = "Can't shoot"@node@"(obstructed)";
		B.RouteGoal = B.FindRandomDest();
		B.MoveTarget = B.RouteCache[0];
		B.SetAttractionState();
		return true;
	}

	B.GoalString = "Heal "$node;
	B.DoRangedAttackOn(node);
	return true;
}

//prepare defensive position status for objective seeking.
//this must be done before objective seeking logic is done.
//here we decide whether or not we want bot to continue seeking a
//defensive position or to directly seek out the objetive.
function EvaluateDefensivePosition(UTBRPlayerManager playerManager)
{
    local UTBot B;
    
    B = playerManager.Bot;

    if (playerManager.CampAtDefensivePosition)
    { 
        if (
            (playerManager.FormationCenter == TheBall) && 
            ((! TheBall.IsControlledByTeamOf(B.Pawn)) || (TheBall.LastHolder == B.Pawn))
           )
        {
            playerManager.CampAtDefensivePosition = false;            
        }    
    
        if ((GetOrders() == 'Attack') && B.LineOfSightTo(TheBall.Position()))
        {
            playerManager.CampAtDefensivePosition = false;
        }     
        
        if (B.Enemy != none)
        {
            playerManager.CampAtDefensivePosition = false;          
        }
        
        if (GetBall(B.Pawn) != none)
        {
            playerManager.CampAtDefensivePosition = false;          
        }
    } 
            
  
    if ((B.DefensivePosition != none) && playerManager.ReachedTarget(B.DefensivePosition) && (! playerManager.CampAtDefensivePosition))
    {
        //allow runner to fall back until they reach fallback point.
        //happens when pathing fails.
        //when runner reaches position, then trying pathing to objective again.

        ShowPath(B, "Reached defensive position");
        
        B.DefensivePosition = none;
        playerManager.SuppressDefensivePositionResetZ = 999999999;            
                    
        if (playerManager.SeekDirect && (! playerManager.ReachedFallback))
        {
            ShowPath(B, "Reached fallback position");
            playerManager.SetReachedFallBack();            
        }         
 
        if (playerManager.SeekDirect)
        {
           //in some cases trying to path to objective is better than using special nav. 
           //use the toggle to make that happen as a secondary resort.
           //generally in this situation pathing is broken but in a few cases it can
           //start working and help.
           //in other cases pathing actually hurts. pathing can break from far away but 
           //then from defensive position it can starting working again but the route
           //is bad and can actually lead bot away from objective.
           //this is why the default for the toggle is to use special navigation. 
               
           if (playerManager.TrySpecialNavigation(playerManager.FormationCenterTarget, true))
           {
               return;         
           }
        }    
    }
      
                  
    if (playerManager.ForceDefensivePosition)
    { 
        //forced fallback after failed path attempt, to keep pathing variety
        playerManager.ForceDefensivePosition = false;
        B.DefensivePosition = FindDefensivePosition(playerManager);      
    }
    
    if (B.DefensivePosition == none)
    {
        playerManager.CampAtDefensivePosition = false;
        playerManager.ForceCamp = false;            
    }
}

/*
Go to some place near to objective/formation center.

formation center is an important concept.
we don't want to just route a bot to a goal.
if goal has a kill zone then they'll jump in goal and die.
also we need the bot to cover not just the goal, but the surrounding
area.        

we don't use bot Defending state or WanderOrCamp for this because it has problems.
bot wouldn't go all the way to Defense point, wouldn't camp enough, etc.
tendency of bot would be to run back and forth between just a couple of places.
with UTSquadAI's ai we find defenders frequently straying from the
formation area, but this code will make sure they hang around there.

here we have control over the process and can do what we want.
*/  
function bool SeekDefensivePosition(UTBRPlayerManager playerManager)
{
    local UTBot B;
    
    B = playerManager.Bot;
                                                               
    if (
        ((B.DefensivePosition != none) && 
          playerManager.ReachedTarget(B.DefensivePosition) && 
          playerManager.CampAtDefensivePosition)
        || playerManager.ForceCamp
        )
    { 
       //camp. use bot Defending state for this as it behaves ok for that purpose.
       
       if (playerManager.ForceCamp)
       {
           ShowPath(B, "Reset defensive position. Forced camp.");        
           B.DefensivePosition = none;
           playerManager.ForceCamp = false;
       }
       
       if ((GetOrders() == 'Defend') || (UTHoldSpot(B.DefensePoint) != None) || (PlayerController(SquadLeader) != None))
       {
           B.CampTime = 5;              
       }
       else
       {
           B.CampTime = 2;        
       }
       
       B.GotoState('Defending','Pausing');
       playerManager.CheckSoak = false;        
       playerManager.CampAtDefensivePosition = false;      
    }
    else
    {
       if ((B.DefensivePosition == none) || playerManager.ReachedTarget(B.DefensivePosition))
       {               
           playerManager.CampAtDefensivePosition = true;
           B.DefensivePosition = FindDefensivePosition(playerManager);
       }

       ShowPath(B, "Go to defensive position " $ B.DefensivePosition);
       
       if (! FindPathToObjective(B, B.DefensivePosition))
       {
          //for some reason actor unreachable, so try another place
           ShowPath(B, "Reset defensive position. Unpathable.");          
           B.DefensivePosition = none;
           playerManager.DefensivePositionFailed = true;
           playerManager.CampAtDefensivePosition = false;

           if (playerManager.SeekDirect)
           {
               playerManager.TrySpecialNavigation(playerManager.FormationCenterTarget, true);
           }
           
           if (! playerManager.SpecialNavigating)
           {                           
               Retask(B);
           }
           
           //though path failed, we still return true to stop ut code from sending bot on a pickup hunt
       }            
       else
       {
           playerManager.DefensivePositionFailed = false;       
       }
    }
        
    return true;
}

function actor FormationCenter(Controller C)
{
    local UTBot B;
    local UTBRPlayerManager playerManager;
    local int i;
    local UTOnslaughtNodeObjective node;
    local UTWeapon w;


    B = UTBot(C);
    
    if ((B == none) || (B.Pawn == none))
    {
        return none;
    }
    
    playerManager = GetPlayerManager(B);
    
    if (playerManager == none)
    {
        return none;
    }
    
    if (playerManager.DoingBallReset) 
    {
        if (GetBall(B.Pawn) != none)
        {
            if (FriendlyGoalInfo.BallResetTarget == FriendlyGoalInfo.KillZone)
            {
                //ut will usually have problems pathing to a killzone so use closest place
                return FindClosestFloorPoint(FriendlyGoalInfo.KillZone);
            }          
            
            return FriendlyGoalInfo.BallResetTarget;
        }
        else
        {
            return TheBall;
        }
    }

    //make bot keep attending to the node it's currently working on
    if (UTOnslaughtNodeObjective(playerManager.FormationCenter) != none)
    {
        if (NodeNeedsAttention(UTOnslaughtNodeObjective(playerManager.FormationCenter), playerManager))
        {
            return playerManager.FormationCenter;
        }
        
        //must restore ratings when done with node due to hack in TellBotHowToHeal()
        ForEach playerManager.Bot.Pawn.InvManager.InventoryActors(class'UTWeapon', w)
        {
            if (w.CanHeal(playerManager.FormationCenter))
            {
           	    w.AIRating = w.default.AIRating;
            }
    	}
        
        playerManager.FormationCenter = none;
    }
        
    for (i = 0; i < UTBRGame(WorldInfo.Game).ObjectiveInfo.Length; i++)
    {   
        node = UTOnslaughtNodeObjective(UTBRGame(WorldInfo.Game).ObjectiveInfo[i].Objective);
        
        if (node != none)
        {
            if (HorizontalDistanceBetween(node, C.Pawn) > 4000)
            {
                continue;
            }
            
            if (! NodeNeedsAttention(node, playerManager))
            {
                continue;
            }
               
            //found a node which bot should attend to
            return node;
        }
    }

    if (EvaluateBallAsFormationCenter(playerManager))
    {       
        return TheBall;
    }
        
    if ( UTHoldSpot(B.DefensePoint) != None )
    {
        //'Hold This Position' command
        return B.DefensePoint;
    }
    
    if ((PlayerController(SquadLeader) != None) && (SquadLeader.Pawn != none))
    {
        //'Cover Me' command
        return SquadLeader.Pawn;
    }    

    //this is after Hold and Cover on purpose, to allow players to call over
    //a bot ball carrier        
    if (GetBall(B.Pawn) != none)
    {
        return EnemyGoal;
    }
    
    if (GetOrders() == 'Freelance')
    {
        //available freelancers will cover the runner
        return TheBall;
    }
            
    if (GetOrders() == 'Attack')
    {              
        return EnemyGoal;
    }

    if (GetOrders() == 'Defend')
    {              
        return FriendlyGoal;
    }    

    return TheBall;
}

//return true if but should do something to this node
function bool NodeNeedsAttention(UTOnslaughtNodeObjective node, UTBRPlayerManager playerManager)
{
    local UTBot B;
     
    B = playerManager.Bot;
    
    if (UTBRGame(WorldInfo.Game).Test_NoNodes)
    {
        return false;
    }
    
    if (
        (node.DefenderTeamIndex != B.GetTeamNum()) || 
        (node.Health < (node.DamageCapacity / 2)) ||
        ((node.Health < node.DamageCapacity) && (! node.bIsConstructing))
        )
    {
		if ((node.DefenderTeamIndex == B.GetTeamNum()) && (! playerManager.HasHealingWeapon(node)) && (GetBall(B.pawn) == none))
        {
      	    //needs healing but healing weapon not found. can't do anything for node.
       	    return false;
    	}
    	
    	if ((node.DefenderTeamIndex == (1 - B.GetTeamNum())) && playerManager.NeedWeapon() && (GetBall(B.pawn) == none))
    	{
    	    //need to attack it but has no weapon, so abort
    	    return false;
        }
        
    	return true;
    }
    
    return false;
}

//called by FormationCenter() to see if bot should get ball
function bool EvaluateBallAsFormationCenter(UTBRPlayerManager playerManager)
{
    local UTBot B;
    
    B = playerManager.Bot;
    
    if ((GetBall(B.Pawn) != none) || (TheBall == none))
    {
        return false;
    }
    
    if ((WorldInfo.TimeSeconds - B.ForcedFlagDropTime) <= 20)
    {
        //if ordered to drop ball, leave it alone for a little while
        return false;
    }
    
    if ((UTDeployable(playerManager.Bot.Pawn.Weapon) != none) && (! bool(BRGame.Settings.AllowBallCarrierWeapons)))
    {
        //if has deployable can't switch to another weapon,
        //so shouldn't get ball if can't switch to ball launcher
        return false;
    }
    
    if ((UTHoldSpot(B.DefensePoint) != None) || 
        (PlayerController(SquadLeader) != None))
    {
        //bots covering or holding position should let someone else get ball.
        //they won't leave their assignment even if they have ball.
        return false;
    }    

    if (playerManager.SpecialNavigating &&
        (B.Pawn.Physics == PHYS_Falling) && 
        (playerManager.MoveTarget == EnemyGoal) &&
        TheBall.IsInMotion() && 
        TheBall.IsPossessedByTeamOf(playerManager.Bot.Pawn) &&
        (TheBall.LastHolder != B.Pawn)        
       )
    {
        //if bot is jumping at enemy goal and ball is in flight then keep going to goal, as a teammate
        //could be passing ball to bot so it can score
        return false;
    }
    
    if (TheBall.Resetting)
    {
        //stops bots from jumping after ball into the reset kill zone
        return false;
    }
    
    if ((! TheBall.IsControlledByTeamOf(B.Pawn)) || 
       (TheBall.LastHolder == B.Pawn) ||
       (DistanceFromBall(B.Pawn) < DistanceFromBall(TheBall.LastHolder))
       )
    {
        if ((GetOrders() == 'Defend') && (Team.Size > 1))
        {
            //only get ball if it's within jurisdiction
            if (HorizontalDistanceBetween(TheBall, FriendlyGoal) <= playerManager.MaxFriendlyGoalDefenseDistance)
            {
               return true;
            }
            
            return false;
        }
       
       return true;
    }
    
    return false;
}

//main ai. called from a ut tick event to tell bots what to do.
function bool BRCheckSquadObjectives(UTBot B)
{
    local UTBRPlayerManager playerManager;
    local bool ret;

    playerManager = GetPlayerManager(B);
        
    if (playerManager == none)
    {
        return false;
    }
              
    if (playerManager.DoingBallReset)
    {
        ret = false;
    }
    else
    {
        ret = BRCheckSquadObjectives2(playerManager);
    }
 
    //must correct jump vars after pathing is done.
    //now when bot reaches destination and trys a jump, they
    //have the correct values to know they can attempt the 
    //jump or else they won't even try
    playerManager.SetupSpecialPathAbilities();                 
                                             
    return ret;
}

//called by CheckSquadObjectives
function bool BRCheckSquadObjectives2(UTBRPlayerManager playerManager)
{
    local UTBot B;
    local controller OurRunner;
  
    B = UTBot(playerManager.Controller);
    
    //try to help ut find high places
    playerManager.SetupSpecialPathAbilities();
               
    if ((GetBall(B.Pawn) != none) && (! playerManager.CanUseWeaponsWithBall()))
    {
        //helps bot keep focus on where they are going when has ball, since can't use weapons
        B.Enemy = none;
        return false;
    }
            
    AddTransientCosts(B,1);

    if (
        (! TheBall.IsHeld()) && 
        (UTVehicle(B.pawn) != none) && 
        (DistanceFromBall(B.Pawn) < 500) && 
        (! UTVehicle(B.pawn).bCanCarryFlag)
       )
    {
        B.LeaveVehicle(true);
    }
            
    if (TheBall.IsHeldByTeamOf(B.Pawn))
    {
        OurRunner = TheBall.HolderController;

        // check if should tow runner to base
        if ( (OurRunner != none) && (UTVehicle(OurRunner.Pawn) != None) && 
            (UTVehicle(OurRunner.Pawn).GetTowingVehicle() == B.Pawn) &&
            CheckTowing(B, UTVehicle(B.Pawn)) )
        {
            ////
            ShowPath(B, "Tow runner");
                      
            return true;
        }
    }

    if (! TheBall.IsHeldByTeamOf(B.Pawn))
    {
        if (B.LineOfSightTo(TheBall.Position()))
        {
                if ( (B.Enemy == None) || (GetBall(B.Enemy) == none) )
                {
                    FindNewEnemyFor(B,(B.Enemy != None) && B.LineOfSightTo(B.Enemy));
                }
                
                if ( WorldInfo.TimeSeconds - LastSeeEnemyRunner > 6 )
                {
                    LastSeeEnemyRunner = WorldInfo.TimeSeconds;

                    //a hack to try to make UTVoice.uc play orb message instead of flag message.
                    //no support for just passing the orb message we want.
                    //UTVoice plays orb message if game class is onslaught.
                    WorldInfo.GRI.GameClass = class'UTOnslaughtGame';                   

                    //to say 'enemy orb carrier here', you gotta send 'INCOMING'.
                    //there is no ENEMYORBCARRIERHERE message parameter, and ENEMYFLAGCARRIERHERE will always
                    //send flag message, even with above game class hack.
                    //B.SendMessage(None, 'ENEMYFLAGCARRIERHERE', 14);
                    B.SendMessage(None, 'INCOMING', 25);
                }
        }


    }


    if (
         (TheBall.IsHeldByTeamOf(B.Pawn) && (DistanceFromBall(B.Pawn) < 800)) ||
         ((PlayerController(SquadLeader) != None) && (DistanceBetween(B.Pawn, SquadLeader.Pawn) < 800)) 
       )
    {
        if ( !B.bInitLifeMessage )
        {
            B.bInitLifeMessage = true;
            B.SendMessage(TheBall.HolderPRI, 'GOTYOURBACK', 10);
        }
    }  
 
    //try to help ut find high places
    playerManager.SetupSpecialPathAbilities();


    return false;
}

function bool WanderNearLeader(UTBot B)
{
  return super.WanderNearLeader(B);
}

function bool TellBotToFollow(UTBot B, Controller C)
{
    return super.TellBotToFollow(B, C);
}

function bool IsLowGravity()
{
    return UTBRGame(WorldInfo.Game).IsLowGravity();
}

//return how far away a1 is from a2
function float DistanceBetween(Actor a1, Actor a2)
{
    return UTBRGame(WorldInfo.Game).DistanceBetween(a1, a2);
}

//return how far away a1 is from a2
function float DistanceBetweenPoints(vector v1, vector v2)
{  
    return UTBRGame(WorldInfo.Game).DistanceBetweenPoints(v1, v2);
}

//return how far away enemy goal is from P
function float DistanceFromEnemyGoal(Pawn P)
{
  return vsize(EnemyGoal.Location - P.Location);
}

//return how far away friendly goal is from P
function float DistanceFromFriendlyGoal(Pawn P)
{
  return vsize(FriendlyGoal.Location - P.Location);
}

//return how far away ball is from P
function float DistanceFromBall(Pawn P)
{
  return vsize(TheBall.Position().Location - P.Location);
}

//return how high a1 is over a2, negative if below. 
function float VerticalDistanceBetween(Actor a1, Actor a2)
{
    return UTBRGame(WorldInfo.Game).VerticalDistanceBetween(a1, a2);    
}

//return the horizontal distance a1 is away from a2.
function float HorizontalDistanceBetween(Actor a1, Actor a2)
{   
    return UTBRGame(WorldInfo.Game).HorizontalDistanceBetween(a1, a2);
}

//return the horizontal distance v1 is away from v2.
function float HorizontalDistanceBetweenPoints(vector v1, vector v2)
{
    return UTBRGame(WorldInfo.Game).HorizontalDistanceBetweenPoints(v1, v2);      
}

//called by ut pathing engine.
//pathing engine will make bot do short detours to pickups along route to objective.
function bool AllowDetourTo(UTBot B,NavigationPoint N)
{
    if (! AllowDetour(GetPlayerManager(B)))
    {
        return false;
    }
    
    return ( N.LastDetourWeight * B.RouteDist > 2 );
}

//called by this class
function bool AllowDetour(UTBRPlayerManager playerManager)
{
    local UTBot B;
    local bool getWeapon;
    
    B = playerManager.Bot;
    
    if (playerManager.DoingBallReset)
    {
        return false;
    }
    
    if ((playerManager.FormationCenter == TheBall) &&
        (HorizontalDistanceBetween(TheBall, FriendlyGoal) < playerManager.LobPassDistance) &&
        (playerManager.HorizontalDistanceFromFormationCenter <= 2500))
    {
        //if ball is held by enemy bot is useless without weapon
        //if ball is not held then even without weapon bot can go get ball and
        //move it out of danger
        getWeapon = TheBall.IsHeldByEnemyOf(B.Pawn) && playerManager.NeedWeapon();
        
        if (playerManager.TakingDefensivePathTime != 0)
        {
           //let bot always get weapon just after spawning
           getWeapon = true;
        }
       
        if ((! getWeapon) && (! TheBall.IsControlledByOtherTeamMember(B.Pawn)))
        {
            //situation is critical. enemy could score. no detours!
                        
            return false;
        }
    }
    
    if ((GetBall(B.Pawn) != none) && (playerManager.HorizontalDistanceFromFriendlyGoal <= playerManager.LobPassDistance))
    {
        return false;
    }
            
    if ((playerManager.FormationCenter == EnemyGoal) &&
        (playerManager.HorizontalDistanceFromFormationCenter < playerManager.MaxFormationCenterDefenseDistance) && 
        (TheBall.LastHolder == B.Pawn))
    {
        //stop ball runner bot from shopping for pickups if they are near enemy goal
        return false;
    }  

    return true;
}

function bool ShouldUseAlternatePaths()
{
    return true;
}

function SetAlternatePathTo(NavigationPoint NewRouteObjective, UTBot RouteMaker)
{
    local UTBot M;

    // override updating route objective so we can switch route cache
    if (NewRouteObjective != RouteObjective && FriendlyGoal != None && EnemyGoal != None)
    {
        // save routes for current objective
        if (RouteObjective == FriendlyGoal)
        {
            FriendlyGoalRoutes = SquadRoutes;
        }
        else if (RouteObjective == EnemyGoal)
        {
            EnemyGoalRoutes = SquadRoutes;
        }
        // restore routes for new objective
        if (NewRouteObjective == FriendlyGoal)
        {
            SquadRoutes = FriendlyGoalRoutes;
        }
        else if (NewRouteObjective == EnemyGoal)
        {
            SquadRoutes = EnemyGoalRoutes;
        }
        else
        {
            SquadRoutes.length = 0;
        }
        SquadRouteIteration = SquadRoutes.length % MaxSquadRoutes;
        RouteObjective = NewRouteObjective;
        // re-enable squad routes for any bots that had it disabled for the old objective
        for (M = SquadMembers; M != None; M = M.NextSquadMember)
        {
            M.bUsingSquadRoute = true;
        }
        PendingSquadRouteMaker = RouteMaker;
    }

    Super.SetAlternatePathTo(NewRouteObjective, RouteMaker);
}

/* BeDevious()
return true if bot should use guile in hunting opponent (more expensive)
*/
function bool BeDevious(Pawn Enemy)
{
    return false;
}

//find closest place at floor level nearest O
function Actor FindClosestFloorPoint(Actor O)
{
    local NavigationPoint N;
    local Actor best;
    local float bestDist, dist;
    local Vector startLoc;

    //find floor point above/below O
    
    startLoc = O.Location;
    dist = BRGame.BlockedDist(O, vect(0,0,-1), 4000);
    bestDist = 999999999;

    if (dist > 0)
    {
        startLoc.Z -= dist;
    }
    
    if ((NavigationPoint(O) != none) || (UTBRBall(O) != none) || (Pawn(O) != none))
    {
        if (DistanceBetweenPoints(O.Location, startLoc) <= 150)
        {
            //already close to floor and is an actor which is pathable to.
            //this is a good optimization rather than spinning through all the nodes below.
            return O;
        }
    }
     
    //find nearest nav point to floor point   
    foreach WorldInfo.RadiusNavigationPoints(class'NavigationPoint', N, startLoc, 4000)
    {
        dist = DistanceBetweenPoints(N.Location, startLoc);
               
        if (dist < bestDist)
        {
            best = N;
            bestDist = dist;
        }
    }
       
    return best;
}

function UTBRPlayerManager GetPlayerManager(UTBot B)
{
    return UTBRGame(WorldInfo.Game).GetPlayerManager(B.PlayerReplicationInfo);
}

/*
    (see UTBRPlayerManager class for documentation on pathing theory used in BR)
    
    see if bot should continue on current path or get a new one.
    return true if continuing on current path and should not get a new one.
    
    retask is called so many times and it doesn't always mean bot should
    do something else or repath. it can be called specifically to make bot rethink
    it's route, it can be called just because ball changes hands, when
    a movement timer runs out, etc.
    
    we can't just blindly let the bot repath because if bot is in route
    somewhere and between path nodes, calling FindPathToObjective or some
    other pathing routine again can upset the bot's route, creating ugly
    situations like looping, running back to a node already visited, totally
    changing route, etc.
    
    if bot is going from A to B and we let a repath get called, it can lose B
    and go somewhere else, even back to A.
    so if retasking keeps getting done, bot can keep
    losing it's movetarget and get in all kinds of strange loops.
        
    basic idea of pathing is a bot starts out from a node to another node and
    retask is done when it reaches target node. if retask is done between nodes
    route can be upset.
    
    so we must smartly detect when a bot truely needs to repath, and otherwise
    just let bot continue on current path undisturbed. 
*/
function bool ContinuePath(UTBRPlayerManager playerManager, Actor newPathTarget)
{
    local UTBot B;

    if (playerManager == none)
    {
        return false;
    }
    
    if (BRGame.Test_AlwaysRepath)
    {
        return false;
    }
       
    B = playerManager.Bot;
    
   
    if (playerManager.ForceRepath)
    {
        playerManager.ForceRepath = false;
        return false;
    }
       
    if (playerManager.PathTarget != newPathTarget)
    {       
        if (
            playerManager.SpecialNavigating && 
            (B.Pawn.Physics == PHYS_Falling)
            )
        {
            if (
                 (
                  ((GetBall(newPathTarget) != none) && GetBall(newPathTarget).IsInMotion()) || 
                  ((newPathTarget == EnemyGoal) && (GetBall(B.Pawn) != none) && (! playerManager.LandedOnBallRejection(newPathTarget)))
                 ) &&
                 (HorizontalDistanceBetween(B.Pawn, newPathTarget) <= 2000)
               )
            {
                //allow bot to reroute mid air if picked up the ball or if chasing moving ball.
                
                playerManager.SetPathTarget(newPathTarget, true);              
                ShowPath(B, "Reroute special navigation mid air to " $ newPathTarget); 
                return true;
            }
        }
   
        return false;
    }
       
    if (playerManager.MoveTarget != none)
    {
        if (playerManager.SpecialNavigating)
        {
            ShowPath(B, "Continuing special navigation to " $ playerManager.MoveTarget);
            return true;
        } 
                      
        if (playerManager.ReachedTarget(playerManager.MoveTarget))
        {
            return false;
        }
        else 
        {         
            //bot continues on current path without repathing.
             
            //this should allow for more stable pathing when path doesn't have to
            //change but retask is being called a lot due to things like ball being
            //dropped and picked up a lot.
            
            //in playerManager.AssignSquadResponsibility2, it sets ForceRepath if Retask was not called,
            //so that ut can refresh path info whenever it needs to, so we will not get here when it's just ut
            //making another pathing call. it was found problems occured if we did not do this. bots would miss
            //their movetarget sometimes. 
            
           
           
            //found that calling SetAttractionState without calling FindPathtowards can have bad effect such as
            //bot missing movetarget, so just let bot continue unless movetimer has run out
            if (B.MoveTimer <= 0)
            {
                playerManager.SetUTBotMoveTarget();                 
                B.SetAttractionState();
            }
    
            ShowPath(B, "Continuing navigation to " $ B.MoveTarget $ " on route to " $ playerManager.PathTarget);
                    
            return true;
        }
    }
    
    return false;
}

/* FindPathToObjective()
Find path a bot should use moving towards objective

pathing breaks down if goal is high in the air.
here are some situations which happen when goal is high in the air. 
tested with GodLike bot because they path better than less skilled bots.
tested with just standard ut functionality, not using BR's navigation functionality.
standard ut pathing was found to be very buggy, so BR navigation routines
are used to overcome these flaws.

whenever FindPathToObjective returns false, MoveTarget is none.
found when using same tests with bot of Average skill, he always breaks down
like in the reg grav case no matter what the gravity is.
so, average bots can not do smart pathing things like a godlike does, such
as path to a ledge nearer to goal.
unfortunately, you can't use B.bSoaking to test for this, as it was always false.

FindBestPathToward() and FindPathToward() are supposed to find a near node to unreachable 
actors but they do not work.

ut3 is supposed to be able to find high places, but it as of epic patch 1.2 it has a bug.
hopefully epic will fix this and we can remove these hacks.
 
regular gravity: 
   pathing works until they get close to goal and then FindPathToObjective is false and MoveTarget is none.
   it keeps reporting this until you kill bot.
   then when bot gets back to the ball which is in the spot where bot died,
   bot will repath to some other location and then break down again as above.
   MoveTarget never gets set to the goal.
   Tried to reset the RouteCache and could not figure out how.
   Would seem you need to kill the bot to get it to repath properly.
   Even so, that would be useless as the MoveTarget never gets set to the goal.
   
low gravity (goal still out of reach):
   pathing rarely breaks. FindPathToObjective generally returns true, 
   sometimes, but rarely, is false.
   however, bot still stands still soaking.
   MoveTarget is pointing to either the goal or some other nav point near to goal.
   pathing properly points bot to the nav point which is best to reach goal from,
   such as a ledge, as long as that place is reachable.    
   However, even when pointing to such a near nav point, and bot can reach that place,
   bot will just stand there soaking because he knows even if he gets there
   he can't reach goal from there.

   
low gravity quad jump (goal in reach from floor):
   pathing never breaks.
   bot sometimes will soak but this is rare.
   bot will attempt to get to the nav point which is closest to goal, such as a ledge,
   and then from there bot will attempt to jump to goal.
   however, the bot can't quad jump so he only double jumps and still doesn't
   reach goal. the second part of the jump may be enough to reach the goal,
   but the bot does not use any steering so if he's even a little off in the 
   initial jump, he will not hit the goal.

*/
function bool FindPathToObjective(UTBot B, Actor O)
{ 
    local bool ret, noPath, ok, forceSpecialNav;
    local UTBRPlayerManager playerManager;
    local float oldBoost, jDist, dist, routeTargetDist, oDist;
    local Actor dest, target;
    local Vector temp;
    local bool doSpecialNav, canJumpToGoal, checkFixedPath, allowLoop;
    local bool usingSpecialJumpPad;
    local int i;
    local UTBRObjectiveInfo info;
    local Actor routeTarget;
    local BRNodeInfoStruct approachInfo, outerAssaultInfo, assaultInfo, nodeInfo;
    local RouteInfoStruct newMoveTargetInfo;
    local string s;
    local Actor nextTarget;

  
    playerManager = GetPlayerManager(B);
    
    if ((playerManager == none) || (B == none) || (B.Pawn == none))
    {
        return false;
    }
    
    if (ContinuePath(playerManager, O))
    {
        return true;
    }
       
    nextTarget = none;
    if (B.RouteCache.Length >= 2)
    {
        nextTarget = B.RouteCache[1];
    }

    routeTarget = playerManager.GetRouteTarget(O);
    routeTargetDist = playerManager.RouteDist(routeTarget);
    oDist = playerManager.RouteDist(O);    
        
    dest = O;
    s = "FindPathToObjective to " $ dest $ "    Dist to " $ routeTarget $ "=" $ routeTargetDist;
    if (O != routeTarget)
    {
        s = s $ "  Dist to " $ O $ "=" $ oDist;
    }
    ShowPath(B, "");  //helps for readability
    ShowPath(B, s);
      
    playerManager.SetPathTarget(O);
    
    info = UTBRGame(WorldInfo.Game).GetObjectiveInfo(routeTarget);    
    
    if (playerManager.IsFormationCenter(O))
    {
        playerManager.TriedDefensivePosition = false;
    }    
            
    // don't ever use gather points when have ball
    if (GetBall(B.Pawn) != none)
    {
        B.bFinalStretch = true;
    }    
   
    if ((GetBall(B.Pawn) != none) && playerManager.OldHasTranslocator && (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons)))
    {   
        //unfortunately one must override the translocator vars every path request or ball
        //carrier will path to strange places. only way to get it to work correctly is to
        //do this before every path request.
        //even stranger is that it only starting worked when NextTranslocTime was set,
        //but also need to set the other vars.
        //stranger still is that for a short while bot still paths like it has the trans, but then
        //stops doing that as it draws closer to objective.
        B.bHasTranslocator = false;
        B.NextTranslocTime = WorldInfo.TimeSeconds + 999999999.0;          
        B.bAllowedToTranslocate = false;
    }

    //update route history
    playerManager.ReachedTarget(playerManager.MoveTarget);
    
    if (playerManager.ReachedTarget(playerManager.MoveTarget))
    {
        if (UTBRPathNode(playerManager.MoveTarget) != none) 
        {       
            if (playerManager.MoveTarget == playerManager.PathSubTarget)
            {
                newMoveTargetInfo = playerManager.BRNodeRouteInfo;
            }
            else if (playerManager.SeekDirect)
            {
                newMoveTargetInfo = UTBRPathNode(playerManager.MoveTarget).GetRouteInfo(routeTarget);
            }
        }
        
        if (playerManager.MoveTarget == playerManager.PathSubTarget)
        {
            ShowPath(B, "Reached path sub target " $ playerManager.PathSubTarget);
            playerManager.SetPathSubTarget(none);
        }
    }
          
    ok = true;
    
    if (! playerManager.SeekDirect)
    {
        //if bot only seeking defensive position and near to it or in defensive area,
        //then don't use assault/approach paths
        
        if (oDist <= 4000)
        {
            ok = false;
        }
        
        if (DistanceBetween(B.Pawn, playerManager.DefensiveCenterTarget) <= playerManager.MaxFormationCenterDefenseDistance)
        {
            ok = false;
        }
        
        if (! playerManager.CanUseAssaultPaths)
        {
            ok = false;
        }
    }
             
    //choose a BRPathNode near to bot which can get bot closer to objective.
    //in this manner a mapper can have make paths which bot prefers to follow.        
    if (ok &&
       (playerManager.FailPathTo != O) &&
       (playerManager.FailedTarget != O) && 
       (newMoveTargetInfo.MoveTarget == none) && 
       (info != none) && (playerManager.PathSubTarget == none))
    {          
        approachInfo.Dist = 999999999;
        assaultInfo.Dist = 999999999;
        outerAssaultInfo.Dist = 999999999;        

        //find nearest assault and approach node
        for (i = 0; i < info.BRPathNodes.Length; i++)
        {
            nodeInfo.Node = info.BRPathNodes[i];                      
            nodeInfo.RouteInfo = UTBRPathNode(nodeInfo.Node).GetRouteInfo(routeTarget);
            
            if (playerManager.ReachedTarget(nodeInfo.Node) && (nodeInfo.RouteInfo.MoveTarget == none))
            {
                //already at end of the path
                continue;
            }
            
            if (nodeInfo.routeInfo.ApproachPath) 
            {
                //it's an optimization to have BRNodeRouteDist call in the if blocks because it's a more
                //expensive operation
                nodeInfo.Dist = playerManager.BRNodeRouteDist(UTBRPathNode(nodeInfo.Node), nodeInfo.RouteInfo);
                
                if (nodeInfo.Dist < approachInfo.Dist)
                {
                    approachInfo = nodeInfo;
                }
            }
            
            if (nodeInfo.RouteInfo.AssaultPath)
            {
                if (HorizontalDistanceBetween(nodeInfo.Node, routeTarget) < HorizontalDistanceBetween(B.Pawn, routeTarget))
                {
                    nodeInfo.Dist = playerManager.BRNodeRouteDist(UTBRPathNode(nodeInfo.Node), nodeInfo.RouteInfo);
                                       
                    if (nodeInfo.Dist < assaultInfo.Dist)
                    {
                        assaultInfo = nodeInfo;
                    }                
                }
                else if (nodeInfo.RouteInfo.CanUseIfBotCloserToRouteTarget)
                {
                    //these assault paths can be used even if bot is closer to target.
                    //they should be normal or heavier weighted paths, not shortened,
                    //to avoid looping back to path. used to resolve bad pathing issues
                    //close to target
                    
                    nodeInfo.Dist = playerManager.BRNodeRouteDist(UTBRPathNode(nodeInfo.Node), nodeInfo.RouteInfo);
                                    
                    if (nodeInfo.Dist < outerAssaultInfo.Dist)
                    {
                        outerAssaultInfo = nodeInfo;
                    }                   
                }
            }
        }
        
        if ((outerAssaultInfo.Node != none) && (assaultInfo.Node == none))
        {
            //use a path further out if allowed and no regular assault path exists
            assaultInfo = outerAssaultInfo;
        }
                
        s = "";
        
        if (assaultInfo.Node != none)
        {
            assaultInfo.Dist = playerManager.BRNodeRouteDist(UTBRPathNode(assaultInfo.Node), assaultInfo.RouteInfo);
            
            s = "Assault path " $ assaultInfo.Node $ " Dist=" $ assaultInfo.Dist;
        }

        if (approachInfo.Node != none)
        {
            //aproach node always sought before going direct to formation center.        
            //we always use approach node even if path to it is broken.
            //if path broken then special navigation will be used.
            
            approachInfo.Dist = playerManager.BRNodeRouteDist(UTBRPathNode(approachInfo.Node), approachInfo.RouteInfo);
            
            if (s != "")
            {
                s = s $ "  ";
            }
            s = s $ "Approach path " $ approachInfo.Node $ " Dist=" $ approachInfo.Dist;                     
        }
        
        if (s != "")
        {
            ShowPath(B, "Examining " $ s);
        }
        
        if (assaultInfo.Dist >= BRGame.BAD_ROUTE_DIST)
        {
            //never use assault path is path to it is broken
            assaultInfo.Node = none;
        }
        
        if (! playerManager.SeekDirect)
        {
             //if bot seeking defensive position, use defensive position path if shorter
             
             if (oDist < approachInfo.Dist)
             {
                 approachInfo.Node = none;
             }
             
             if (oDist < assaultInfo.Dist)
             {
                 assaultInfo.Node = none;
             }             
        }        
        
        target = none;
        
        if (assaultInfo.Node != none)
        {
            dist = routeTargetDist;
            if (approachInfo.Node != none)
            {
                dist = approachInfo.Dist;
            }
                                                         
            if (assaultInfo.Dist < dist)
            {
                //assault node used when it gives a better route
                target = UTBRPathNode(assaultInfo.Node).FindPathEntryPoint(assaultInfo.RouteInfo, playerManager);                
            }
        }
        
        if ((target == none) && (approachInfo.Node != none))
        {
            target = UTBRPathNode(approachInfo.Node).FindPathEntryPoint(approachInfo.RouteInfo, playerManager);
        }
        
        if (target != none)
        {
            dest = target;
        }
    }
             
    if ((newMoveTargetInfo.MoveTarget == none) &&
        (UTBRPathNode(playerManager.PathSubTarget) != none) &&
        (playerManager.BRNodeRouteInfo.RouteTarget == routeTarget) &&
        (playerManager.BRNodeRouteInfo.ReachedRadius != 0) &&
        (DistanceBetween(B.Pawn, UTBRPathNode(playerManager.PathSubTarget)) <= 
         playerManager.BRNodeRouteInfo.ReachedRadius))
    {
        ShowPath(B, "Entered into ReachedRadius " $ playerManager.BRNodeRouteInfo.ReachedRadius $ " of " $ UTBRPathNode(playerManager.PathSubTarget));
        
        newMoveTargetInfo = playerManager.BRNodeRouteInfo;
    }

    if ((! playerManager.SeekDirect) && (newMoveTargetInfo.MoveTarget == routeTarget))
    {
        //bot not seeking objective so disallow movetarget to objective
        newMoveTargetInfo.MoveTarget = none;
    }  
    
    if (newMoveTargetInfo.MoveTarget != none)
    {
        s = "pathing";
        if (newMoveTargetInfo.ForceToMoveTarget)
        {
            forceSpecialNav = true;
            s = "forcing";
        }
            
        playerManager.SetPathSubTarget(newMoveTargetInfo.MoveTarget);
        dest = playerManager.PathSubTarget;        
        
        if (UTBRPathNode(dest) != none)
        {
            playerManager.BRNodeRouteInfo = UTBRPathNode(dest).GetRouteInfo(routeTarget);
        }       
       
        ShowPath(B, "BRPathNode " $ s $ " bot MoveTarget to " $ dest);
    }            
          
    if (playerManager.PathSubTarget != none)
    {
        if (UTBRPathNode(playerManager.PathSubTarget) != none)
        {
            dist = playerManager.BRNodeRouteDist(UTBRPathNode(playerManager.PathSubTarget), playerManager.BRNodeRouteInfo);        
        }   
        else
        {
            dist = playerManager.RouteDist(playerManager.PathSubTarget);        
        }      

         
        dest = playerManager.PathSubTarget;
        s = "Subrouting bot to " $ dest $ " Route dist=" $ dist;
        
        if (playerManager.BRNodeRouteInfo.RouteTarget == routeTarget)
        {
            if (playerManager.BRNodeRouteInfo.ReachedRadius != 0)
            {
                s = s $ "  ReachedRadius=" $ playerManager.BRNodeRouteInfo.ReachedRadius;
            }
            
            if (playerManager.BRNodeRouteInfo.MoveTarget != none)
            {
                s = s $ "  Next MoveTarget=" $ playerManager.BRNodeRouteInfo.MoveTarget;
            }
        }
                 
        ShowPath(B, s);
    }

    if ((! playerManager.SeekDirect) && (playerManager.PathSubTarget == none))
    {
        //once bot has decided to go directly to DefensivePosition, disallow
        //assault paths to keep bot on path, else bot could get into loops
        playerManager.CanUseAssaultPaths = false;
    }    
   
    
    /*
    //this shows that epic pathing can't find high places properly.
    //the results are None False False False False False
    SendDebugMessage("FindPathToward=" $ B.FindPathToward(EnemyGoal,true));
    SendDebugMessage("FindPathToObjective=" $ Super.FindPathToObjective(B, EnemyGoal));    
    SendDebugMessage("FindBestPathToward1=" $ B.FindBestPathToward(EnemyGoal, false, true));
    SendDebugMessage("StartMoveToward1=" $ B.StartMoveToward(EnemyGoal));
    SendDebugMessage("FindBestPathToward2=" $ B.FindBestPathToward(EnemyGoal, true, true));
    SendDebugMessage("StartMoveToward2=" $ B.StartMoveToward(EnemyGoal));
    */


    playerManager.SetupSpecialPathAbilities();
    
    if (forceSpecialNav)
    {
        //used for when a BRPathNode forces MoveTarget using special navigation
        ret = true;
    }  
    else
    {
        ret = false;
        
        if ((UTBRBall(O) != none) && (UTBRBall(O).Holder != none))
        {       
            //if ball is held then block holder's way to home goal
            if (TryIntercept(playerManager, UTBRBall(O).Holder, FriendlyGoal))
            {
                ret = true;
            }
        }

        if (! ret)
        {
            ret = Super.FindPathToObjective(B, dest);
        }
    }
    
    noPath = ! ret;
    
    if (UTBRBall(dest) != none)
    {
        playerManager.NoBallPath = ! ret;
    }
    
    if (UTBRGoal(dest) != none)
    {
        playerManager.NoGoalPath = false;           
    }  

    checkFixedPath = ((! ret) && (! UTBRGame(WorldInfo.Game).Test_NoHackPath) && (UTBRGoal(dest) != none) && (UTPawn(B.Pawn) != none));

    if (checkFixedPath && playerManager.CanJumpToHeightOf(dest))
    {
        checkFixedPath = false;
        ShowPath(B, "Not checking for fixed path. Goal Reachable.");
    }

    if (checkFixedPath)                  
    {   
        //ut3 was not able to find the goal, probably because it's too high
        //it's hard to apply this fix for every place a bot might path to, such as a weapon pickup
        //real high up, because SetHackJumpPathing must be called before the path request, and the path
        //request then resets the jump vars.
        
        //we only do this for goals. the pathing which results from this is ultimately not navigatable
        //but for high up goals will end up with the bot jumping from a good spot at goal. then if
        //bot doesn't reach via jump it can shoot ball at goal.
        //this kind of situation wouldn't apply to any other actor.
        //normally this wouldn't be needed for friendly goal, but bot will want to path to that
        //goal when trying to reset ball if a killzone is behind the goal.

        playerManager.NoGoalPath = true;

        //first, override jumps to make bot think he can get there, and try pathing again
        oldBoost = UTPawn(B.Pawn).MultiJumpBoost;
        SetHackJumpPathing(B);
        ret = Super.FindPathToObjective(B, dest);
        UTPawn(B.Pawn).MultiJumpBoost = oldBoost; //must restore value because it affects actual jump physics
        
        playerManager.SetupSpecialPathAbilities();
        
        ShowPath(B, "Find path for broken pathing to " $ dest $ ". Result: " $ ret);        
               
        //pathing can still fail do to more ut bugs. 
        //doesn't always happen, but sometimes when starting the next match, 
        //SetHackJumpPathing pathing will fail for the entire match, but then next match it'll be okay.
        
        if (ret)
        {
            //because jump height is hacked to force path to high goal to work, there could nodes in this
            //path which bot can not get to. so basically we are evaluating our own reachspecs here.
            //unfortunately even in low grav quad jump and the goal is totally reachable from any spot,
            //ut3 pathing can still fail even though our fixed reachspec evaluation here does not fail.
            //note: sometimes the path in routecache is not complete so final goal may not be in cache yet
            temp = B.Pawn.Location;
            for (i = 0; i < B.RouteCache.Length; i++)
            {                           
                if (! playerManager.CanJumpToHeightOf(B.RouteCache[i], temp.Z))
                {
                    ShowPath(B, "Rejected fixed path. Failed jump check.");                  
                    ret = false;
                    break;
                }   
                else if (UTBRGoal(B.RouteCache[i]) != none) 
                {
                    canJumpToGoal = true;                
                }        
                
                temp = B.RouteCache[i].Location;                
            }
        }
        
        
        //see if should try to use jumppad which is pointing at goal.
        //if it does not look like bot can reach goal by jumping then try the jumppad.
        //unfortunately even with a jumppad pointing right at goal ut pathing can break down 
        //and won't even path to the pad so we must force the bot to go to pad.
        
        if (playerManager.CanJumpTo(dest))
        {
            canJumpToGoal = true;
        }
        
        if (playerManager.CanJumpToHeightOf(dest) && (HorizontalDistanceBetween(B.Pawn, dest) <= 1000))
        {
            canJumpToGoal = true;
        }

        if (
            (info != none) &&
            (info.JumpPad != none) &&
            (GetBall(B.pawn) != none) && 
            (! UTBRGame(WorldInfo.Game).Test_NoGoalJumpPad) &&
            (! canJumpToGoal)
           )
        {
            dest = info.JumpPad;
            usingSpecialJumpPad = true;
            playerManager.SetupSpecialPathAbilities();            
            ret = Super.FindPathToObjective(B, dest);
            ShowPath(B, "Find path to special goal jumppad to " $ dest $ ". Result: " $ ret);
        }         
       
        if (! ret)
        {
            ShowPath(B, "No fixed goal path");
            
            //if path to goal jump pad failed, then pathed jump below will be tried to goal
            dest = O;        
        }
        
    }
    
    
    if (playerManager.FailPathTo == O)
    {
        //path is being aborted so that alternate navigation, such as special
        //navigation, can be used
        ShowPath(B, "Force failing path to " $ O);
        ret = false;
    }
    playerManager.FailPathTo = none; 
     
                   
    if (ret)
    {
        //sometimes ut will path a bot to some destination through a br goal.
        //this is not good. bot can get stuck or die.
        //examples: teleporters are behind each goal and bot tries to use home goal teleporter to reach enemy goal.
        //          ball falls very near to goal instead of getting ball bot goes into goal and gets stuck.
        
        for (i = 0; i < B.RouteCache.Length; i++)
        {
            if ((B.RouteCache[i] != none) &&
                (FriendlyGoalInfo.JumpPad == B.RouteCache[i]) &&
                (playerManager.FormationCenter == EnemyGoal) &&
                ((O == EnemyGoal) || (O == B.DefensivePosition)))
            {
                //another unsavory situation. bot trying to get to enemy goal jumps on 
                //friendgoal jumppad into goal and dies. seen it happen on TR-Sanctuary.
                //however, can't rule out all paths with this pad in it because
                //if ball is near this jumppad the jumppad is often in the route cache but bot
                //usually gets MoveTarget set to ball before it reaches the pad so this is ok.
                
                ret = false;
                ShowPath(B, "Rejected path because trying to path through friendly goal jump pad to enemy goal. Failed cache: ");
                break;
            }
            
            if ((UTBRGoal(B.RouteCache[i]) != none) && (B.RouteCache[i] != dest))
            {
                ret = false;
                ShowPath(B, "Rejected path because invalid node " $ B.RouteCache[i] $ " is found in it which is not the destination. Failed cache: ");
                break;
            } 
        }
        
        if ((! ret) && UTBRGame(WorldInfo.Game).Test_ShowPath)
        {
            ShowRouteCache(playerManager, true);
        }
    }     


    /* 
      check for bot going back to node already visited.
      in ut3 bots can loop back and forth between two nodes and get in other 
      strange infinite loops.
               
      must be careful about detecting loops because for example a bot
      could get knocked off a ledge and have to repeat a path, but that's not a loop.
      we prevent that from begin falsly detected as loop by checking to make sure
      bot has reached it's current MoveTarget. if knocked off a ledge, it won't
      have reached it.
           
      have seen a bot go down a path part way, and as soon as it starts along that
      path, ut3 makes a new path back the way it came. once it goes down that path,
      ut3 then makes a new path back the other way, and thus bot gets in infinite 
      loop.
      sometimes ut3 will keep routing bot back to the same reached path nodes and
      bot gets stuck there. 
      there are so many buggy pathing situations in ut3.
      
      allow loop if one node was not reached and other reached.
      example: bot is going to A and hits a rock so ut
      then routes bot to B and then to A. even though the path was A B A
      this is not a looping condition because the first A was not reached.     
    */

    if (
        (B.MoveTarget != none) &&
        (UTPickupFactory(B.MoveTarget) != none) &&
        (nextTarget != B.MoveTarget) &&
        (! playerManager.IsInRouteHistory(B.MoveTarget))
       )
    {
        //bot will often detour to a pickup and then go back to path it was on,
        //so must detect this and clear history so loop will not be detected
        ShowPath(B, "Taking detour to pickup");
        playerManager.SetAllowRevist(B.MoveTarget);
    }
           
    allowLoop = false;
    if ((UTOnslaughtNodeObjective(O) != none) && (playerManager.MoveTarget == O))
    {
        //allow loop if O is power node, as bot will run back and forth while healing
        allowLoop = true;
    }
            
    if (
        (! BRGame.Test_NoLoopChecks) &&
        ret &&
        (! playerManager.SuppressLoopChecks) &&
        (! allowLoop) &&
        playerManager.IsReachedNode(playerManager.MoveTarget, 0) &&
        (B.MoveTarget != O) &&
        playerManager.DisallowRevisit(B.MoveTarget)
       ) 
    {
       //force bot to next unvisited node on route, if none found, fail path 
                  
        ret = false;
      
        for (i = 0; i < B.RouteCache.Length; i++)
        {
            if (! playerManager.HasReached(B.RouteCache[i]))
            {
                ret = true;
                dest = B.RouteCache[i]; 
                ShowPath(B, "Forcing bot to " $ dest $ " because going back to " $ B.MoveTarget);
                B.MoveTarget = dest;
                
                //a blockage could be screwing up pathing so use special nav
                forceSpecialNav = true;
                
                break;                
            } 
        }
        
        if (! ret)
        {
            ShowPath(B, "Failing path because going back to " $ B.MoveTarget);
            if (BRGame.Test_ShowPath)
            {
                ShowRouteCache(playerManager);
            }  
        }
    } 

    if (
        (! BRGame.Test_NoLoopChecks) &&    
        ret &&
        (playerManager.MoveTarget != none) &&
        (! playerManager.SuppressLoopChecks) &&
        playerManager.IsFailedNode(playerManager.MoveTarget, 0) &&         
        (playerManager.IsReachedNode(B.MoveTarget, 1) ||
         playerManager.IsReachedNode(B.MoveTarget, 2)) &&
        (B.MoveTarget != playerManager.MoveTarget)
       )
    {
        //force bot to reach failed target instead of going back to prior reached node.
        //this is better than just failing the path because of looping because
        //the forced navigation may be successful and then bot can continue on path.

        ShowPath(B, "Forcing bot to " $ playerManager.MoveTarget $ " because failed to reach and bot is going back to " $ B.MoveTarget);
        dest = playerManager.MoveTarget;
        B.MoveTarget = dest;
        
        //a blockage could be screwing up pathing so use special nav
        forceSpecialNav = true;
    }
    
    if (
        (! BRGame.Test_NoLoopChecks) &&
        ret &&
        (! playerManager.SuppressLoopChecks) &&
        playerManager.IsFailedNode(playerManager.MoveTarget, 0) &&
        playerManager.IsFailedNode(B.MoveTarget, 1) 
       )
    {
        //stop bot looping between two failed nodes
        
        ShowPath(B, "Failing path because it's repeating between two failed nodes. Bot was going back to " $ B.MoveTarget);
        ret = false;
    } 
    
    playerManager.SuppressLoopChecks = false;      
       
    if (! ret)
    {
        ShowPath(B, "FindPathToObjective failed");
        
        if (usingSpecialJumpPad)
        {
            //if special navigation kicks in we want dest to be the objective
            dest = O;
        }
    }  
    

    //evaluate if bot should use special navigation to reach destination.
    //especially when pathing is failing, sometimes using special navigation
    //is the only way to reach it.
    
    doSpecialNav = false;
  
    ok = false;
    
    if (noPath && (UTBRGoal(dest) != none))
    {
        //ut will make bot jump at goal as long as goal isn't too high.
        //bot hitting a blockage like the rim or getting to edge of a dropoff will trigger the jump.
        //we don't want to override that behavior so setting jDist to a reasonable height which ut 
        //probably would not jump for.    
        ok = true;
        jDist = 200;        
    }    
    
    if ((UTBRBall(dest) != none) && playerManager.CloseEnoughToJump(dest)) 
    {
        ok = true;
            
        //ball can get lodged a little above bot's head and ut won't make it jump    
        jDist = 60;
    }
      
    //bot in final approach to target. make a jump if needed.
    if (
        ret && ok &&
        ((dest.Location.Z - B.Pawn.Location.Z) > jDist) &&
        (B.MoveTarget == dest)
       )
    {    
        doSpecialNav = true;
        forceSpecialNav = true;
        ShowPath(B, "Forcing jump in FindPathToObjective in final approach");
    }  
    
    if (! ret)
    {
        //trying a jump or other special navigation when pathing fails solves a lot of issues.
        //for example, bot could be on a high platform and then try to get
        //to a place below but pathing fails (as will usually happen when
        //moving down from high places). initiating a series of jumps to
        //the target will get bot down and to the target.
        
        doSpecialNav = true;
    }
    
    if (
        doSpecialNav && (GetBall(dest) != none) && 
        GetBall(dest).IsInMotion() && playerManager.JumpingPrefered(dest)
       )
    {
        //when chasing ball and it's in motion don't jump after it here.
        //let CheckObjectiveJump make the decision based on random number.  
        doSpecialNav = false;
    }    
    
    if ((UTBRGame(WorldInfo.Game).Test_NavigateWithBall) && (GetBall(B.Pawn) != none))
    {
        //for test. bot immediately uses special navigation.
        doSpecialNav = true;
        playerManager.SetReachedFallBack();
        ret = false;
    }
   
    if (forceSpecialNav)
    {
        doSpecialNav = true;
    }
    
    if ((! ret) && playerManager.SeekDirect && (B.DefensivePosition == O))
    {
        //when seeking objective and pathing fails we get as close as we can
        //using the defensive position and then from there use special navigation.
        //using special navigation to reach defensive position would not make sense
        //because we are counting on having a good pathable place to reach first before
        //using the navigation 
        doSpecialNav = false;
    }
       
    if ((O != playerManager.FormationCenter) && (playerManager.FailedTarget == O))
    {
        //for non formation center targets, DefensivePosition is not used, so 
        //attempt special navigation once, and after that allow path to fail
        doSpecialNav = false;
        ret = false;
        ShowPath(B, "Rejecting path to " $ O $ " because prior navigation failed");
    }
    
    if ((! ret) && playerManager.GettingSpecialItem() && doSpecialNav && playerManager.TriedForcedSpecialNavigationToPathTarget)
    {
        //only allow forced special navigation to special item once, then fail it out and have bot do something else
        ShowPath(B, "Not special navigating to special item because already tried");
        doSpecialNav = false;
    }
   
    if (doSpecialNav)
    {
        playerManager.TriedPath = true;
    }
    
    if (doSpecialNav && playerManager.TrySpecialNavigation(dest, forceSpecialNav))
    {
        ret = true;
    }    
        
    playerManager.SetupSpecialPathAbilities();
              
    if (! ret)
    {
        playerManager.FailedTarget = O;
        playerManager.SetPathSubTarget(none);
    }
    else
    {
        playerManager.FailedTarget = none;
        
        if (B.MoveTarget != none)
        {
            playerManager.SetMoveTarget(B.MoveTarget);
        }
        
        if ((B.Pawn.Physics == PHYS_Walking) && (UTBRBall(B.MoveTarget) != none) && 
            (GetBall(B.Pawn) == none) && playerManager.ReachedTarget(B.MoveTarget))
        {
            //needed to stop bot standing by ball without picking it up.
            //happens when bot is moving and bot is following closly, when
            //ball comes to rest bot would just stand there.
            
            ShowPath(B, "Forcing bot to ball");
            B.SetAttractionState();
            
            B.Pawn.Velocity = Normal(B.MoveTarget.Location - B.Pawn.Location);
            B.Pawn.Velocity.Z = 0;
            B.Pawn.Velocity = normal(B.Pawn.Velocity);
            B.Pawn.Velocity = B.Pawn.Velocity * B.Pawn.GroundSpeed;
                    
            B.Pawn.Acceleration = Normal(B.MoveTarget.Location - B.Pawn.Location); 
            B.Pawn.Acceleration.Z = 0;
            B.Pawn.Acceleration = normal(B.Pawn.Acceleration);
            B.Pawn.Acceleration = B.Pawn.Acceleration * B.Pawn.AccelRate;
        }        
    }

    return ret;
}


//ut3 has great trouble finding high places.
//here we override the jumps to a high value to make bot think it
//can reach that place. then it will at least navigate near to the
//place. this override is done only for a pathing operation and
//must be done before each path operation.
//UTBot.SetupSpecialPathAbilities is called frequently which resets
//the jump vars.
//note a super high value is done here because even if the bot can't actually
//make the jump, we still want the bot to get as near as possible to the place.
function SetHackJumpPathing(UTBot B)
{
    UTPawn(B.Pawn).MultiJumpBoost = 99999;
}

function bool AllowTranslocationBy(UTBot B)
{
    return ( GetBall(B.Pawn) == none );
}

function bool TryToIntercept(UTBot B, Pawn P, Actor RouteGoal)
{
    //don't allow ut to use this. we need control over the process.
    return false;
}

//br version of UTSquadAI.TryToIntercept.
//this is called from FindPathToObjective (where it ought to be in ut3 code) so that
//the core pathing logic has control.
function bool TryIntercept(UTBRPlayerManager playerManager, Pawn P, Actor RouteGoal)
{
    local UTBot B;
    
    B = playerManager.Bot;
       
	if ( (P == B.Enemy) && B.Pawn.RecommendLongRangedAttack() && (P != None) && B.LineOfSightTo(P) )
	{
	    ShowPath(B, "Long range fight ball runner");
		B.FightEnemy(false,0);
		return true;
	}

	B.MoveTarget = None;
	if ( B.ActorReachable(P) )
	{
		if ( B.Enemy != P )
			SetEnemy(B,P);
		if ( B.Enemy != None )
		{
            ShowPath(B, "Fight ball runner");
			B.FightEnemy(true,0);
			return true;
		}
    }
  
    //this is a very disappointing function. it only works if a UTPawn is passed. 
    //tried passing a Scout but that didn't work.
    //so we can't use this function for defensive bot routes unless ball is held.
	B.MoveTarget = B.FindPathToIntercept(P,RouteGoal,true);
	
	if ( B.MoveTarget != None )
	{
        if (B.Pawn.ReachedDestination(B.MoveTarget))
        {
		    return false;
        }
        
	    if (B.StartMoveToward(P))
        {
      	    ShowPath(B, "Using intercept path to ball runner");
            return true;
        }        	
	}
	
	return false;
}

function bool VisibleToEnemiesOf(Actor A, UTBot B)
{
    if ( (B.Enemy != None) && FastTrace(A.Location, B.Enemy.Location + B.Enemy.GetCollisionHeight() * vect(0,0,1)) )
        return true;
    return false;
}

function bool CheckVehicle(UTBot B)
{
    if (UTVehicle(B.Pawn) != none)
    {
        //ut uses this for soak checks but it's laughable. hardly ever works
        //and when on some chance it does it just gets in the way of our own
        //soak checks so just turn it off
        UTVehicle(B.Pawn).StuckCount = 0;
    }
    
    if (B.RouteGoal != B.MoveTarget || Vehicle(B.RouteGoal) == None) // so bot will get in obstructing vehicle to drive it out of the way
    {
        if ( (TheBall.Holder == None) && (DistanceFromBall(B.Pawn) < 1600) )
            return false;
        if ( (GetBall(B.Pawn) != none) && (DistanceFromEnemyGoal(B.Pawn) < 1600) )
            return false;
    }

    return Super.CheckVehicle(B);
}

function UTPlayerReplicationInfo GetPlayerReplicationInfo(Pawn P)
{
    return UTBRGame(WorldInfo.Game).GetPlayerReplicationInfo(P);
}

function UTBRBall GetBall(Actor actor)
{
    return UTBRGame(WorldInfo.Game).GetBall(actor);
}

function bool MustKeepEnemy(Pawn E)
{
    if ( (E != None) && (GetPlayerReplicationInfo(E) != None) && (GetBall(E) != none) && (E.Health > 0) )
        return true;
    return false;
}

function bool NearEnemyBase(UTBot B)
{
    return EnemyGoal.BotNearObjective(B);
}

function bool NearHomeBase(UTBot B)
{
    return FriendlyGoal.BotNearObjective(B);
}

function bool BallNearOurBase()
{
    return ( VSize(TheBall.Position().Location - FriendlyGoal.Location) < FriendlyGoal.BaseRadius );
}

function bool OverrideFollowPlayer(UTBot B)
{
    return false;
}

//navigates bot along a path which is between home goal and ball.
//this way they can attack along a path which is defending their goal.
function bool GotoDefensivePath(UTBRPlayerManager playerManager)
{
    local vector targetSpot;
    local NavigationPoint N, best;
    local float bestDist, dist;
    local UTBot B;
    
    B = UTBot(playerManager.Controller);    

    targetSpot = FriendlyGoal.Location + (Normal(TheBall.Location - FriendlyGoal.Location) * DistanceBetween(FriendlyGoal, B.Pawn));
    
    foreach WorldInfo.RadiusNavigationPoints(class'NavigationPoint', N, FriendlyGoal.Location, vsize(FriendlyGoal.Location - targetSpot))
    {    
        dist = vsize(targetSpot - N.Location);
        if ((dist < bestDist) || (best == none))
        {
            best = N;
            bestDist = dist;
        }
    }
    
    ShowPath(B, "Go along defensive path to " $ best);
    
    if ((best != none) && FindPathToObjective(B, best))
    {
        return true;
    }
    
    return false;
}

//goto vehicle using br pathing
function bool GotoVehicle(UTVehicle SquadVehicle, UTBot B)
{
    local Actor BestEntry;
    local bool ret;
    local UTBRPlayerManager playerManager;

    //fixed for br. if you were in a vehicle bots would go up to you and stand there.
    if (SquadVehicle.Driver != none)
    {
       return false;
    }      
    
    BestEntry = SquadVehicle.GetMoveTargetFor(B.Pawn);

    playerManager = GetPlayerManager(B);
    if (playerManager == none)
    {
        return false;
    }
    
    if (playerManager.ReachedTarget(BestEntry))
    {
        if (Vehicle(B.Pawn) != None)
        {
            B.LeaveVehicle(true);
            return true;
        }
        B.EnterVehicle(SquadVehicle);
        return true;
    }

    //need to use BR pathing to find vehicle. makes sure bot stays on route to vehicle
    //and uses br special pathing. in regular ut bots can loop back and forth or just
    //stand still trying to reach vehicle.
    
    ret = FindPathToObjective(B, BestEntry);
    
    if (ret)
    {
        B.RouteGoal = SquadVehicle;
        SquadVehicle.SetReservation(B);
    }
    
    return ret;
    
    /*
    if ( B.ActorReachable(BestEntry) )
    {
        if (Vehicle(B.Pawn) != None)
        {
            B.LeaveVehicle(true);
            return true;
        }
        B.RouteGoal = SquadVehicle;
        B.MoveTarget = BestEntry;
        SquadVehicle.SetReservation(B);
        B.GoalString = "Go to vehicle 1 "$BestEntry;
        B.SetAttractionState();
        return true;
    }

    BestPath = B.FindPathToward(BestEntry,B.Pawn.bCanPickupInventory);
    if ( BestPath != None )
    {
        B.RouteGoal = SquadVehicle;
        SquadVehicle.SetReservation(B);
        B.MoveTarget = BestPath;
        B.GoalString = "Go to vehicle 2 "$BestPath;
        B.SetAttractionState();
        return true;
    }
    */

/*
not sure what this code could possibly accomplish.
above code already has verified BestEntry is not reachable so
trying to continue on here makes the bot just stand there....

    if ( (VSize(BestEntry.Location - B.Pawn.Location) < 1200)
        && B.LineOfSightTo(BestEntry) )
    {
        if (Vehicle(B.Pawn) != None)
        {
            B.LeaveVehicle(true);
            return true;
        }
        B.RouteGoal = SquadVehicle;
        SquadVehicle.SetReservation(B);
        B.MoveTarget = BestEntry;
        B.GoalString = "Go to vehicle 3 "$BestEntry;
        B.SetAttractionState();
        return true;
    }
*/

    //return false;
}

function float VehicleDesireability(UTVehicle V, UTBot B)
{
    local float ret;
    local UTBRPlayerManager playerManager;

    //unfortunately ut stills sometimes tries to get a vehicle even if we have
    //set vehiclelosttime so we must override that annoying behavior here
    if (V.VehicleLostTime > WorldInfo.TimeSeconds)
    {
        return 0;
    } 
    
    playerManager = GetPlayerManager(B);
     
    if ((playerManager != none) && (V == playerManager.FailedTarget))
    {
        //bot already failed to reach so don't try again
        return 0;
    }    
        
    ret = VehicleDesireability2(V, B); 

    //in standard ut if two vehicles are nearby and the higher rated one is not pathable then
    //bot will get no vehicles. below code fixes that by giving bad rating to
    //vehicle with bad path
   
    if ((playerManager != none) && (! CanSpecialReach(playerManager, V)))
    {
        ret = 0;
    }
    
    return ret;
}

//return true if bot is able to reach target by pathing or possible special move 
function bool CanSpecialReach(UTBRPlayerManager playerManager, Actor target)
{
    playerManager.SetupSpecialPathAbilities();

    if (playerManager.Bot.FindPathToward(target) == none)  
    {
        if ((! playerManager.CanJumpToHeightOf(target)) || (! playerManager.Bot.LineOfSightTo(target)))
        {
            return false;
        }
    }
    
    return true;
}

//fixed from base class to allow bots to use all vehicles
//return a value indicating how useful this vehicle is to the bot
function float VehicleDesireability2(UTVehicle V, UTBot B)
{
    local float result;

    // if bot has the flag and the vehicle can't carry flags, ignore it
    if ( B.PlayerReplicationInfo.bHasFlag && !V.bCanCarryFlag )
    {
        return 0;
    }
    // if we're attacking and vehicle is meant for defenders (or vice versa), ignore it
    if (V.AIPurpose != AIP_Any)
    {
        //fixed this for BR
        //used to return 0. should return a low rating instead.
        //so what would happen is a defending bot wouldn't use a Scavenger sitting right by it 
        //for defense, which isn't too smart.
        if (CurrentOrders == 'Defend')
        {
            if (V.AIPurpose != AIP_Defensive)
            {
                return 0.01;
            }
        }
        else if (V.AIPurpose != AIP_Offensive)
        {
            if ((CurrentOrders == 'Attack') && (V.bIsOnTrack))
            {
                return 0.0;
            }
            
            return 0.01;
        }
    }
    // if vehicle is low on health and there's an enemy nearby (so don't have time to heal it), ignore it
    if (V.Health < V.HealthMax * 0.125 && B.Enemy != None && B.LineOfSightTo(B.Enemy))
    {
        return 0;
    }
    // otherwise, let vehicle rate itself
    result = V.BotDesireability(self, (Team != None) ? Team.TeamIndex : 255, SquadObjective);
    if ( V.SpokenFor(B) )
    {
        return result * V.ReservationCostMultiplier(B.Pawn);
    }
    else
    {
        return result;
    }
}

function BallTakenByOurTeam(Controller C)
{
    local UTBot M;

    if ( (PlayerController(SquadLeader) == None) && (SquadLeader != C) )
        SetLeader(C);

    if (UTBot(C) != None)
    {
        UTBot(C).Pawn.SetAnchor(EnemyGoal);
        SetAlternatePathTo(EnemyGoal, UTBot(C));
    }

    for ( M=SquadMembers; M!=None; M=M.NextSquadMember )
        if (M.MoveTarget == TheBall)
            M.MoveTimer = FMin(M.MoveTimer,0.05 + 0.15 * FRand());
}

function bool AllowTaunt(UTBot B)
{
    return ( (FRand() < 0.5) && (PriorityObjective(B) < 1));
}

function bool ShouldDeferTo(Controller C)
{
    if (GetBall(C.Pawn) != none)
    {
        return true;
    }
    
    return Super.ShouldDeferTo(C);
}

function byte PriorityObjective(UTBot B)
{
    if (GetBall(B.Pawn) != none)
    {
        if ( EnemyGoal.BotNearObjective(B) )
            return 255;
        return 2;
    }

    if ( TheBall.IsHeldByEnemyOf(B.Pawn) )
        return 1;

    return 0;
}

function float ModifyThreat(float current, Pawn NewThreat, bool bThreatVisible, UTBot B)
{
    if ( (GetPlayerReplicationInfo(NewThreat) != None)
        && (GetBall(NewThreat) != none)
        && bThreatVisible )
    {
        if ( (VSize(B.Pawn.Location - NewThreat.Location) < 1500) || (B.Pawn.Weapon != None && UTWeapon(B.Pawn.Weapon).bSniping)
            || (DistanceFromFriendlyGoal(NewThreat) < 2000) )
            return current + 6;
        else
            return current + 1.5;
    }
    else if ( NewThreat.IsHumanControlled() )
        return current + 0.5;
    else
        return current;
}

function Actor GetTowingDestination(UTVehicle Towed)
{
    //return none to make bot just do it's normal task
    return none;
}

function bool MustCompleteOnFoot(Actor O, optional Pawn P)
{
    local UTBRPlayerManager playerManager;
    local UTBot B;

    if ((UTBRBall(O) != none) && (UTVehicle(P) != none) && (! UTVehicle(P).bCanCarryFlag))
    {
        return true;
    } 

    //for now return false as we already have code which checks for this. 
    //found that bots leave vehicles too much if this stock ut code isn't disabled.
    return false;

    
    //rest of code not used at moment ...    
    
    if (UTVehicle(P) == none)
    {
        return false;
    }
        
    B = UTBot(P.Controller);
    
    if (B == none)
    {
        return false;
    }
   
    playerManager = GetPlayerManager(B);
    
    if (playerManager == none)
    {
        return false;
    } 
    
    if (
        UTVehicle(playerManager.Bot.Pawn).bCanFly &&
        (UTBRBall(playerManager.FormationCenter) != none) &&
        (GetBall(playerManager.Bot.Pawn) == none) &&        
        UTBRBall(playerManager.FormationCenter).IsInMotion()
       )
    {
        return false;
    }

    return Super.MustCompleteOnFoot(O, P);
}

function bool AllowContinueOnFoot(UTBot B, UTVehicle V)
{
    local UTBRPlayerManager playerManager;
  
    //for now we just make it false, as we already have other code such as soak checks and special navigation
    //for making sure bot can handle blockages and getting stuck.
    //it's really annoying when bot should go somewhere in vehicle and can make it but instead ditches
    //vehicle due to some difficulty. 
    //exception is for hoverboard. in hoverboard bot will just fall off dropoffs, get stuck easier, go in circles,
    //and other general stupid stuff.
       
    if (UTVehicle_Hoverboard(B.Pawn) != none)
    {
        return Super.AllowContinueOnFoot(B, V);
    }
          
    return false;


    //rest of code not used at moment...
       
    playerManager = GetPlayerManager(B);    
    
   
    //force bot to try to score in vehicle.
    //BR has other custom code which will make bot leave vehicle when needed.
    //exception is hoverboard. they suck at driving those and instead of jumping over
    //a dropoff into goal, would just fall into dropoff.
    if ((GetBall(B.Pawn) != none) && (UTVehicle_Hoverboard(B.Pawn) == none))
    {
        return false;
    }
     
    if ((playerManager != none) &&
        (B.Enemy == none) && 
        (GetBall(B.Pawn) == none) &&
        (playerManager.HorizontalDistanceFromFormationCenter <= playerManager.MaxFormationCenterDefenseDistance) &&
        ((WorldInfo.TimeSeconds - playerManager.MoveTargetTime) < 15)
       )
    {
        //bot would leave vehicle when trying to get to defensive position if it was near a wall.
        //bot will just go back and forth so make them camp. After camping a new position will be selected.

        playerManager.ForceCamp = true;
        return false;
    }

    // not if can cover runner from here
    if ( TheBall.IsHeldByTeamOf(B.Pawn) && (GetBall(B.Pawn) == none) &&
        (B.Enemy == None || GetPlayerReplicationInfo(B.Enemy) == None) &&
        B.LineOfSightTo(TheBall.Position()) )
    {
        return false;
    }
    else
    {
        return Super.AllowContinueOnFoot(B, V);
    }
}

function ModifyAggression(UTBot B, out float Aggression);

//return true if B is heading to enemy goal.
//must search entire routecache, as movetarget and routegoal can be set to points along the way.
function bool HeadingToEnemyGoal(UTBot B)
{
    local int i;

    if ((B.RouteGoal == EnemyGoal) || (B.MoveTarget == EnemyGoal))
    {
        return true;
    }
    
    for (i = 0; i < B.RouteCache.length; i++)
    {
        if (B.RouteCache[i] == EnemyGoal)
        {
            return true;
        }
    }       
    
    return false;
}

//return max distance bot is allowed to be from Center
function float GetMaxDefenseDistanceFrom(Actor Center, UTBot B)
{
    local float dist, temp;

    if (Center == none)
    {
        return 0;
    }
           
    if (UTHoldSpot(Center) != None)
    { 
        //'Hold this position'
        return 2000;
    }
    
    if (Pawn(Center) != None)
    {
        //'Cover me'
        return 2000;
    }
    
    dist = 5000;    
    if (UTBRGoal(Center) != none)
    {
        //some maps are small so make it 3/4 way between goal and ball home base
        temp = (DistanceBetween(Center, TheBall.HomeBase) * 0.75);
        if (temp < dist)
        {
            dist = temp;
        }
    }   
    
    if (Center == EnemyGoal)
    {
        if (GetBall(B.Pawn) != none)
        {
           //sometimes ball carrier must fall back, but don't let them go too far 
           dist = 2000;
        }
                 
        //makes bots form front line with runner as runner approaches goal
        if (TheBall.IsHeldByTeamOf(B.Pawn) &&
           (GetBall(B.Pawn) == none))
        {
           temp = DistanceFromEnemyGoal(TheBall.Holder);
           if (temp < dist)
           {
              dist = temp;
              if (dist < 1500)
              {
                  dist = 1500;
              }
           }
        } 
    }
    
    if (Center == TheBall)
    {
        dist = 2000;
    }
          
    return dist;
}

//base class requires this signature
function float RateDefensivePosition(NavigationPoint N, UTBot CurrentBot, Actor Center)
{
    local UTBRPlayerManager pm;
    
    pm = GetPlayerManager(CurrentBot);
    return RateDefensivePosition2(N, CurrentBot, Center, Center, 0, pm, none, false);
}

function ShowPathRating(UTBRPlayerManager playerManager, Actor A, string msg)
{
    if ((playerManager != none) && (UTBRGame(WorldInfo.Game).Test_ShowPathRatings))
    {
         SendBotDebugMessage(playerManager.Bot, "Path rating: " $ A $ " " $ msg);
    }
}

function float RateDefensivePosition2(NavigationPoint N, UTBot CurrentBot, Actor Center, Actor CenterTarget, float maxDefenseDist, UTBRPlayerManager playerManager, Actor centerNearPoint, bool hasEnoughDefensePoints)
{
    local float Rating;
    local UTBot B;
    local int i;
    local ReachSpec ReverseSpec;
    local bool bNeedSpecialMove;
    local UTPawn P;
    local float dist; 
   

    if (playerManager == none)
    {
        ShowPath(CurrentBot, "playerManager not found in RateDefensivePosition2");
        return -1.0;
    }
    
    /* 
      do not need a trace between N and Center.
      bot should be able to defend from any place near to goal.
      regular navigation points ok to defend from.
      don't allow jump pads or the center to be defense point.
      centers are usually the goal and we don't want defending bot jumping into goal.
      
    if ( N.bDestinationOnly || N.IsA('Teleporter') || N.IsA('PortalMarker') || (N.bFlyingPreferred && !CurrentBot.Pawn.bCanFly) ||
        (!FastTrace(N.Location, Center.GetTargetLocation()) && (!Center.bHasAlternateTargetLocation || !FastTrace(N.Location, Center.GetTargetLocation(, true))))  )
    {
        return -1.0;
    }
    */
    
    if ( 
          N.IsA('Teleporter') || N.IsA('PortalMarker') || (N.bFlyingPreferred && !CurrentBot.Pawn.bCanFly) ||
          (UTJumpPad(N) != none) || (N == Center) || (N == CenterTarget) 
       )
    {
        ShowPathRating(playerManager, N, "Failed portal check");
        return -1.0;
    }  
             

    if (UTBRGoal(N) != none)
    {
        //goals themselves are generally a bad place to defend from.
        //bot will die if it's their goal, or fall into killzone below goal, etc.
        
        ShowPathRating(playerManager, N, "Failed goal check");
                
        return -1;
    }            
    
    if ((UTBRPathNode(N) != none) && UTBRPathNode(N).NotDefensivePosition)
    {
        ShowPathRating(playerManager, N, "Marked as not defensive position");
        return -1;
    } 
    
    if (Caps(N.Tag) == "NOTDEFENSIVEPOSITION")
    {
        ShowPathRating(playerManager, N, "Tagged as not defensive position");
        return -1;        
    }

    if (! playerManager.SeekDirect)
    { 
        //seeking place near to objectve.
        //defending, attacker seeking a stakeout waiting for ball runner, etc
        
        if (hasEnoughDefensePoints)
        {
           if (UTDefensePoint(N) == none)
           {
               //if enough UTDefensePoints are around center then only allow those
               //to be used
               return -1;
           }
        }
        
        if (((N.Location.Z - CurrentBot.Pawn.Location.Z) > 500) &&
            ((CenterTarget.Location.Z - CurrentBot.Pawn.Location.Z) < 500))
        {
           //keep bots away from high places, they tend to get stuck,
           //it's difficult for a bot to defend from there, their
           //mobility would be limited and it would be difficult to 
           //pursue ball runner.
           //however if Center is also high, then let them go high.
           
           ShowPathRating(playerManager, N, "Failed high place check");
                   
           return -1;
        }
        
        // if bot can't translocate or double jump, disregard points only reachable by that method
        P = UTPawn(CurrentBot.Pawn);
        if (!CurrentBot.bAllowedToTranslocate || P == None || !P.bCanDoubleJump)
        {
            bNeedSpecialMove = true;
            for (i = 0; i < N.PathList.length; i++)
            {
                if (N.PathList[i].End.Nav != None)
                {
                    ReverseSpec = N.PathList[i].End.Nav.GetReachSpecTo(N);
                    if ( ReverseSpec != None &&
                        (CurrentBot.bAllowedToTranslocate || !ReverseSpec.IsA('UTTranslocatorReachSpec') || !ReverseSpec.IsBlockedFor(P)) &&
                        ((ReverseSpec.reachFlags & 16) == 0 || (P != None && P.bCanDoubleJump)) )
                    {
                        bNeedSpecialMove = false;
                        break;
                    }
                }
            }
            if (bNeedSpecialMove)
            {
                ShowPathRating(playerManager, N, "Failed special move check");
                return -1.0;
            }
        }
    
        // make sure no squadmate using this point, and adjust rating based on proximity
        Rating = 1;
        for ( B=SquadMembers; B!=None; B=B.NextSquadMember )
        {
            if ( B != CurrentBot )
            {
                if ( (B.DefensePoint == N) || (B.DefensivePosition == N) )
                {
                    ShowPathRating(playerManager, N, "Failed squadmate check");
                    return -1;
                }
                
                //this code is bad. it makes bots cluster in a tight group and stay in one area.
                //rather, they should spread out and cover the whole area.
                //else if (( B.Pawn != None ) && (VSize(B.Pawn.Location - N.Location) <= 800))
                //{
                //    Rating *= 0.002*VSize(B.Pawn.Location - N.Location);
                //}
            }
        }
       
        if ((UTDefensePoint(N) != none) && (rand(5) == 0) && (! hasEnoughDefensePoints))
        {
            //make bot tend to go to defense points more, but not always go to them.
            //it's bad to make them always go to defense points, as that 
            //would make the bots too predictable.
            //players would then always know to lob spam at those points and kill the bot.
            Rating = Rating + 0.10;
        }
        
        if ((maxDefenseDist != 0) && (Center == EnemyGoal) && TheBall.IsHeldByTeamOf(CurrentBot.Pawn) &&
            (! TheBall.IsHeldBy(CurrentBot.Pawn)))    
        {
            //calculate defense perimeter around enemy goal
            
            dist = maxDefenseDist * 0.4;
            
            if (dist < 1000)
            {
                //map too small for a perimeter
                dist = 0;
            }
            
            if (dist > 2500)
            {
                //keep perimeter down to reasonable dist
                dist = 2500;
            }
                    
            //if bot can't see runner then it stays outside the perimeter.
            //to be in good position to help when runner comes into view.
            //if bot can see runner then go inside the perimeter to help clear goal area or get pass.
        
            if ((dist != 0) && (HorizontalDistanceBetween(CenterTarget, N) <= dist))
            {
                if (CurrentBot.LineOfSightTo(TheBall.Position()))
                {            
                    Rating = Rating + 0.10;
                }
                else
                {
                    Rating = Rating - 0.10;                
                }
            }
        }
    }

    if (playerManager.SeekDirect)
    {
        //going directly to objective.
        //seeking ball, ball runner seeking goal, etc    

        if (
            (UTBRGoal(Center) != none) && 
            (GetBall(CurrentBot.Pawn) != none) && 
            (HorizontalDistanceBetween(Center, N) <= 2000) &&
            (playerManager == none)
            )
        {
            //for ball carrier seeking goal rate places nearer goal better 
            Rating = Rating + 0.10;          
        }
             
        /*
        SeekDirect is a very important thing, especially for ball seeking.
        the ball can get in all kinds of unpathable places. having bot go to a
        pathable place near to ball and from there to ball will generally work ok.
        
         we don't check for pathabilty when just defending, as bot could be up in a 
         place where pathing is breaking so we need to allow selection of an unpathable
         defensive position and let bot use special navigation to get there, otherwise
         bot will be stuck in that place.
         
         it's tempting to restrict the search to a tight radius but testing has proved
         this not to work. nodes nearby can actually be worse, and you can end up being restricted 
         to just a few bad nodes. having a wider variety of nodes to choose from is better. 
         bot is more likely to get a good place and has more pathing variety. 
         in many cases nodes further away work just as well as nodes nearby.  
        */
               
        //if jump is potentially possible from N to Center than rate higher
        if (playerManager.CanJumpToHeightOf(Center, N.Location.Z))
        {
           dist = Center.Location.Z - N.Location.Z;
           if ((dist > 0) && (BRGame.BlockedDist(playerManager, vect(0,0,1), dist, N.Location) > 0))
           {
               //area above N is blocked, so not good for jump
               Rating = Rating - 0.10;
           }
           
           Rating = Rating + 0.10;
        }
        
        //rate things on same vertical level better
        //vertical closness is better than horizonal so rate a bit higher if real close vertically.
        //this makes places on the same level as Center tend to be rated bettter, as it doesn't
        //make sense to choose a place on some other floor level.
        //sometimes Center is a high up goal so it doesn't make sense to rate things at that level
        //so centerNearPoint is used as a nearby pathable place
        //which identifies the nearest floor level from which to reach Center.

        if (centerNearPoint == none)
        {
            centerNearPoint = Center;
        }
        
        dist = abs(centerNearPoint.Location.Z - N.Location.Z);

        if (dist <= 200)
        {
            Rating = Rating + 0.30;
        }
        else if (dist <= 400)
        {
            Rating = Rating + 0.06;
        } 
        else if (dist <= 600)
        {
            Rating = Rating + 0.05;
        }                 

        
        if (HorizontalDistanceBetween(Center, N) <= 1000)
        {
            //closer places are more likely to be pathable to Center 
            Rating = Rating + 0.10;
        }
    }
     
    ShowPathRating(playerManager, N, "Rated at " $ Rating);
                        
    return Rating;
}

function NavigationPoint FindDefensivePositionFor(UTBot B)
{
    local UTBRPlayerManager playerManager;
    
    playerManager = GetPlayerManager(B);       
    if (playerManager == none)
    {
        return none;
    }
    
    return FindDefensivePosition(playerManager);
}

//find a place around formation center to defend from or attack from.
//also used to find an alternate route to objective when pathing fails.
function NavigationPoint FindDefensivePosition(UTBRPlayerManager playerManager)
{
    local NavigationPoint N, Best, First;
    local Actor centerNearPoint;
    local float rating;
    local Actor Center, approachNode, CenterTarget;
    local Array<RatingStruct> points;
    local float maxDefenseDist, zdist, temp, dist, bestDist;
    local UTBot B;
    local int i, maxIx, i2;
    local RatingStruct rs;
    local bool ok, hasEnoughDefensePoints;
    local UTBRObjectiveInfo info;
   
    B = playerManager.Bot;
    
    // if on track, just pick random node on that track
    if (UTVehicle(B.Pawn) != None && UTVehicle(B.Pawn).bIsOnTrack)
    {   
        return B.FindRandomDest();
    }

    Center = FormationCenter(B);
    if (Center == None)
    {
        Center = B.Pawn;
    }
    
    CenterTarget = Center;

    if (! playerManager.SeekDirect)
    {    
        info = UTBRGame(WorldInfo.Game).GetObjectiveInfo(Center);
        
        if (info != none)
        {
            hasEnoughDefensePoints = info.HasEnoughDefensePoints;

            if (UTBRGoal(Center) != none)
            {    
                //see if a UTBRPathNode ApproachPath node exists for this objective.
                //if it does we should seek a defensive position around it instead,
                //as the approach path node is meant to be a substitute for the objective
                //for pathing purposes, and the objective is in some out of the way place.
                //example is EWokForrest map where goal is in very long tube and approach
                //node is at mouth of tube.
                        
                bestDist = 999999999;
                
                for (i = 0; i < info.BRPathNodes.Length; i++)
                {
                    for (i2 = 0; i2 < info.BRPathNodes[i].RouteInfo.Length; i2++)
                    {
                        if ((info.BRPathNodes[i].RouteInfo[i2].RouteTarget == Center) &&
                            info.BRPathNodes[i].RouteInfo[i2].ApproachPath)
                        {
                            dist = DistanceBetween(playerManager.Bot, info.BRPathNodes[i]);
                            if (dist < bestDist)
                            {
                                approachNode = info.BRPathNodes[i];
                                bestDist = dist;
                            } 
                        }
                    }
                }
                
                if (approachNode != none)
                {
                   ShowPath(playerManager.Bot, "Using " $ approachNode $ " as defensive center for " $ Center);
                   CenterTarget = approachNode;             
                }
            }
        }
    }  
    
    playerManager.DefensiveCenterTarget = CenterTarget;
    
    
    maxDefenseDist = GetMaxDefenseDistanceFrom(Center, B);
    
    //allow for a high Center is to add a portion of the Z height to the radius
    zdist = CenterTarget.Location.Z - B.Pawn.Location.Z;
    temp = CenterTarget.Location.Z - TheBall.HomeBase.Location.Z;
    if (temp > zdist)
    {
        zdist = temp;
    }
    
    if ( zdist > 500)
    {
        maxDefenseDist = maxDefenseDist + (zdist * 0.6);
    }    
        
    
    if (playerManager.SeekDirect)
    {
        centerNearPoint = FindClosestFloorPoint(CenterTarget);
    }    
    
    foreach WorldInfo.RadiusNavigationPoints(class'NavigationPoint', N, CenterTarget.Location, maxDefenseDist)
    {
        rating = RateDefensivePosition2(N, B, Center, CenterTarget, maxDefenseDist, playerManager, centerNearPoint, hasEnoughDefensePoints);          

        for (i = 0; i <= points.Length; i++)
        {
            if ((i == points.length) || (rating >= points[i].Rating))
            {
                ok = true;
                points.Insert(i, 1);
                
                rs.Target = N;
                rs.Rating = rating;
                 
                points[i] = rs;
                
                break;
            }
        }
    }
    
    //now select a random place among the best rated and test pathability.
    //if place is bad due to unpathability, then throw out and pick another.
    //pathability check is done during this last stage and done as little as
    //possible because it's a more expensive operation

    Best = none;
    First = none;
            
    while ((Best == none) && (points.Length > 0))
    {
        maxIx = 0;
        rating = points[0].Rating;
        for (i = 1; i < points.Length; i++)
        {
            if (points[i].Rating != rating)
            {
                break;
            }
            
            maxIx = i;
        }    
        
        ok = true;
        
        i = rand(maxIx + 1);
        playerManager.SetupSpecialPathAbilities();

        if (First == none)
        {
            First = points[i].Target;
        }
                      
        if (playerManager.Bot.FindPathToward(points[i].Target) == none)
        {
            //when bot is directly seeking objective, never use an unpathable 
            //alternate path since it's being used to resolve pathing problems
                
            ok = false;
            
            if ((! playerManager.SeekDirect) && (UTDefensePoint(points[i].Target) != none))
            {
                //it's possible for a defending bot to get itself into a bad place where
                //most pathing is failing but for a few nearby nodes. we don't want
                //bot just staying around those nodes, so make sure some nodes are 
                //always tried for no matter what. UTDefensePoint is good for this
                //since we know bots should go to them for sure.
                ok = true;
            }
        }  
        
        if (ok)
        {
            Best = points[i].Target;
        }
        else
        {
            points.Remove(i, 1);
        }
    }

    if (Best == none)
    {
        Best = First;
        ShowPath(playerManager.Bot, "Using best unpathable node for defensive position because all others failed");
    }
                     
    return Best;
}

function SendBotDebugMessage(Controller C, string msg)
{
    BRGame.SendDebugMessage(C.PlayerReplicationInfo.GetPlayerAlias() $ " " $ msg);
}

function SendDebugMessage(string msg)
{
    BRGame.SendDebugMessage(msg);
}

//for debugging. show path bot is taking.
function ShowPath(Controller C, string msg)
{
    BRGame.SendPlayerDebugMessage(C, msg);
}

defaultproperties
{
   MaxSquadSize=3
   bShouldUseGatherPoints=True
   Name="Default__UTBRSquadAI"
   ObjectArchetype=UTSquadAI'UTGame.Default__UTSquadAI'
}
